/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        $Id: pgpKeyMan.c,v 1.118.2.15 2001/08/27 19:26:04 hal Exp $
____________________________________________________________________________*/
#include <string.h>

#include "pgpConfig.h"
#include "pgpSDKPriv.h"		/* DO NOT REMOVE */

#include "pgpContext.h"
#include "pgpEncodePriv.h"
#include "pgpEventPriv.h"
#include "pgpDebug.h"
#include "pgpKeyPriv.h"
#include "pgpPubKey.h"
#include "pgpRandomX9_17.h"
#include "pgpRandomPool.h"
#include "pgpSigSpec.h"
#include "pgpStr2Key.h"
#include "pgpKeySpec.h"
#include "pgpHash.h"
#include "pgpEnv.h"
#include "pgpP11Key.h"
#include "pgpTokenLib.h"
#include "pgpRegExp.h"
#include "pgpTimeDate.h"
#include "pgpTrustPriv.h"
#include "pgpUtilitiesPriv.h"
#include "pgpRandomPoolPriv.h"
#include "pgpRnd.h"
#include "pgpSymmetricCipherPriv.h"
#include "pgpX509Priv.h"
#include "pgpPassCach.h"

#define MAXRSABITS		4096

#define elemsof(x) ((unsigned)(sizeof(x)/sizeof(*x)))

/*  INTERNAL FUNCTIONS */

/*
 * Internal functions for certifying a key or userid.
 *
 *  Sigspec holds the information about the kind of signature to make.
 *  It is automatically freed after the sig itself is created, or if there
 *  is an error in trying to modify it.
 */

#define SIG_EXPORTABLE			TRUE
#define SIG_NON_EXPORTABLE		FALSE
#define SIG_EXPORTABLEHASHED	TRUE
#define SIG_EXPORTABLEUNHASHED	FALSE

extern PGPMutex_t sRpcMutex;

static PGPError
sCreateSigSpec(
	PGPContextRef 		context,
	PGPKeyDBObj		   *signer,
	PGPByte 			sigtype,
	char const 			*passphrase,
	PGPSize 			passphraseLength,
	PGPBoolean 			hashedPhrase,
	PGPUInt32			cacheTimeOut,
	PGPBoolean			cacheGlobal,
	PGPSigSpec		  **psigspec)
{
    PGPSigSpec		*sigspec = NULL;
	PGPEnv			*pgpEnv;
	PGPError		error = kPGPError_NoErr;

	pgpEnv = pgpContextGetEnvironment( context );

	/* Error if not enough entropy for a safe signature */
	if( ! PGPGlobalRandomPoolHasMinimumEntropy() )
		return kPGPError_OutOfEntropy;
	
	if (IsntNull(passphrase) && passphraseLength == 0)
		passphrase = NULL;

	if (!signer || !pgpKeyIsSec (signer) ||
		!(pgpKeyUse (signer) & PGP_PKUSE_SIGN))
	    return kPGPError_SecretKeyNotFound;

	if( !pgpSecPassphraseOK( signer, (PGPByte *) passphrase,
						passphraseLength, hashedPhrase, cacheTimeOut,
						cacheGlobal  ) )
		return kPGPError_BadPassphrase;
    sigspec = pgpSigSpecCreate (pgpEnv, signer, sigtype);
    if (!sigspec)
	    return kPGPError_OutOfMemory;
	pgpSigSpecSetPassphrase( sigspec, (PGPByte *) passphrase,
					passphraseLength, hashedPhrase );

	*psigspec = sigspec;
	return error;
}

	static PGPError
sSigSpecAddRAK(
	PGPSigSpec			*sigspec,
	PGPKeySet			*rakset,
	PGPUInt32			rakclass)
{
	PGPKeyDBObj		*krkey;
	PGPKeyIter		*rakIter;
	PGPError		err = kPGPError_NoErr;

	/* Handle revocation authorizations */
	pgpAssert( IsntNull( rakset ) );

	err = PGPNewKeyIterFromKeySet( rakset, &rakIter );
	if (IsPGPError( err )) {
		pgpSigSpecDestroy (sigspec);
		return err;
	}
	while( IsntPGPError( PGPKeyIterNextKeyDBObj( rakIter,
				kPGPKeyDBObjType_Key, &krkey ) ) ) {
		PGPByte krinfo[22];
		PGPByte pkalg;

		/* Note that rakclass must have 0x80 set to be effective */
		pgpKeyID8 (krkey, &pkalg, NULL);
		krinfo[0] = rakclass;
		krinfo[1] = pkalg;
		pgpKeyFingerprint20 (krkey, krinfo+2);
		err = pgpSigSpecSetRevocationKey (sigspec, 0, krinfo,
										  sizeof(krinfo) );
		if (IsPGPError(err)) {
			pgpSigSpecDestroy (sigspec);
			PGPFreeKeyIter (rakIter);
			return err;
		}
	}
	PGPFreeKeyIter (rakIter);
	/* Make this signature non-revocable */
	pgpSigSpecSetRevocable (sigspec, 0, FALSE);

	return kPGPError_NoErr;
}

	static PGPError
sSigSpecSetExportability(
	PGPSigSpec			*sigspec,
	PGPBoolean 			exportable,
	PGPBoolean 			exportableHashed)
{
	if (!exportable) {
		pgpSigSpecSetExportable (sigspec,
							 (exportableHashed ? 0 : kPGPSigFlags_Unhashed),
							 exportable);
	}
	return kPGPError_NoErr;
}


static PGPError
sSigSpecSetTimes(
	PGPSigSpec			*sigspec,
	PGPTime				sigCreation,
	PGPUInt32 			sigExpiration)
{
	PGPContextRef	 context;
	PGPEnv			*pgpEnv;

	context = pgpSigSpecContext( sigspec );
	pgpEnv = pgpContextGetEnvironment( context );

	if( sigCreation != 0 )
		pgpSigSpecSetTimestamp( sigspec, sigCreation +
				(60 * 60 * pgpenvGetInt(pgpEnv, PGPENV_TZFIX,  NULL, NULL)));
	
	if (sigExpiration)
		pgpSigSpecSetSigExpiration (sigspec, 0, sigExpiration);

	return kPGPError_NoErr;
}


static PGPError
sSigSpecSetTrustParams(
	PGPSigSpec		   *sigspec,
	PGPKeyDB		   *kdb,
	PGPByte 			trustDepth,
	PGPByte 			trustValue,
	char const 			*sRegExp)
{
	PGPEnv				*pgpEnv;
	PGPContextRef		context;

	if( IsntNull( sRegExp ) )
		pgpSigSpecSetRegExp (sigspec, 0, sRegExp);

	context = PGPPeekKeyDBContext( kdb );
	pgpEnv = pgpContextGetEnvironment( context );

	/* Ignore trustValue for ordinary level 0 signatures */
	if (trustDepth != 0) {
		/* Convert trust value to extern format */
		if (trustValue != 0)
			trustValue = pgpTrustOldToExtern(pgpEnv, trustValue);
		/* Note that setting nonzero trustvalue forces V4 sigs */
		pgpSigSpecSetTrustLevel (sigspec, 0, trustDepth, trustValue);
	}

	return kPGPError_NoErr;
}

/* Create the certification based on info in sigspec */
static PGPError
sCertifyObject(
	PGPSigSpec		   *sigspec,
	PGPKeyDBObj		   *signee)
{
	PGPContextRef		context;
	PGPBoolean		 	selfsig = FALSE;
	PGPKeyDBObj *	 	parent;
 	PGPKeyDBObj *	  	signer;
	PGPByte			 	sigtype;
	PGPInt32			pkalg;
    PGPError		 	error = kPGPError_NoErr;

	context = pgpSigSpecContext( sigspec );
	sigtype = pgpSigSpecSigType( sigspec );
	signer = pgpSigSpecSeckey( sigspec );
	pgpGetKeyNumber( signer, kPGPKeyProperty_AlgorithmID, &pkalg );

	for( parent = signee; !OBJISTOPKEY(parent); parent = parent->up ) {
		if( parent == signer ) {
			selfsig = TRUE;
			break;
		}
	}

	if (pkalg > kPGPPublicKeyAlgorithm_RSA &&
			sigtype == PGP_SIGTYPE_KEY_GENERIC && selfsig) {
		/* Propagate sig subpacket information */
		PGPByte const *p;
		PGPSize plen;
		PGPBoolean hashed;
		pgpSigSpecSetVersion (sigspec, PGPVERSION_4);
		if ((p=pgpKeyFindSubpacket (signer,
				SIGSUB_PREFERRED_ENCRYPTION_ALGS, 0,
				&plen, NULL, &hashed, NULL, NULL, NULL)) != 0
			&& hashed) {
			pgpSigSpecSetPrefAlgs (sigspec, 0, p, plen);
		}
		if (pgpKeyExpiration (signer)) {
			PGPUInt32 period = pgpKeyExpiration (signer) -
				pgpKeyCreation (signer);
			pgpSigSpecSetKeyExpiration (sigspec, 0, period);
		}
	}
	
	/* Due to a bug in 5.0, all sigs directly on keys must be version 2_6.
	 * However the only signatures 5.0 handles directly on keys are key
	 * revocations.
	 */
	if (pgpObjectType( signee ) == RINGTYPE_KEY &&
			!pgpKeyIsSubkey( signee )  &&
			sigtype == PGP_SIGTYPE_KEY_REVOKE ) {
		pgpSigSpecSetVersion( sigspec, PGPVERSION_3 );
	}

	/* Perform the signature calculation */
	error = pgpSignObject (signee, sigspec);
	pgpKeyDBChanged( PGPPeekKeyDBObjKeyDB(signee), TRUE );

	pgpSigSpecDestroy (sigspec);

	return error;
}



/*  Check for a 'dead' key.  A dead key is revoked or expired. 
	There's not much you can do with such a key. */

	static PGPError
pgpKeyDeadCheck( PGPKeyDBObjRef	key)
{
    PGPBoolean	revoked, expired;
    PGPError	err;
	
	err	= pgpGetKeyBoolean (key, kPGPKeyProperty_IsRevoked, &revoked);
	if ( IsntPGPError( err ) && revoked )
		err	= kPGPError_KeyRevoked;
	
	if ( IsntPGPError( err ) )
	{
		err	= pgpGetKeyBoolean (key, kPGPKeyProperty_IsExpired, &expired);
		if ( IsntPGPError( err ) && expired )
			err	= kPGPError_KeyExpired;
	}
	
	return ( err );
}


/* Same for subkey... */

static PGPBoolean
pgpSubKeyIsDead (PGPKeyDBObjRef subkey)
{
    PGPBoolean   revoked, expired;
	
	pgpGetSubKeyBoolean (subkey, kPGPSubKeyProperty_IsRevoked, &revoked);
	pgpGetSubKeyBoolean (subkey, kPGPSubKeyProperty_IsExpired, &expired);
	return (revoked || expired);
}


#if 0
/*  Find the default private key.  Get the name (or keyid) from the 
	environment, and find the PGPKeyDBObj.  If there is no default 
	key defined in the environment, return NULL unless there is 
	only one private key in the key database.
	*/

	static PGPError
pgpGetDefaultPrivateKeyInternal(
	PGPKeyDBRef	keyDB,
	PGPKeyDBObj **	outKey)
{
	PGPError			err			= kPGPError_NoErr;
	PGPByte *			keyIDData	= NULL;
	void *				vkeyIDData;
	PGPSize				keyIDSize	= 0;
	PGPContextRef		context		= PGPPeekKeyDBContext( keyDB );
	
	PGPValidatePtr( outKey );
	*outKey	= kInvalidPGPKeyDBObjRef;
	
	err	= PGPsdkPrefGetData( context, kPGPsdkPref_DefaultKeyID,
				&vkeyIDData, &keyIDSize );
	keyIDData = vkeyIDData;
	if ( IsntPGPError( err ) )
	{
		PGPKeyID		keyID;
		
		err	= PGPImportKeyID( keyIDData, &keyID );
		if ( IsntPGPError( err ) )
		{
			err	= PGPGetKeyByKeyID( keyDB, &keyID,
						kPGPPublicKeyAlgorithm_Invalid, outKey );
		}
		
		/* we used public API call; must free using PGPFreeData() */
		PGPFreeData( keyIDData );
	}
	
	return err;
}
#endif

static PGPInt32
sGetKeyTokenNum( PGPKeyDBObj *key )  
{
	PGPKeyInfo *kinfo;

	if( !pgpKeyIsOnToken(key) )
		return -1;

	pgpAssert(OBJISKEY(key));

	kinfo = pgpKeyToKeyInfo( key );

	return kinfo->tokenNum1 - 1;
}

/*  END OF INTERNAL FUNCTIONS */



/* This is also called on the backend for revoking subkeys */
	PGPError
pgpRevokeKey_internal (
	PGPKeyDBObjRef		key,
	char const *		passphrase,
	PGPSize				passphraseLength,
	PGPBoolean			hashedPhrase,
	PGPUInt32			cacheTimeOut,
	PGPBoolean			cacheGlobal
	)
{
    PGPKeyDBRef			 keys = NULL;
	PGPContextRef		 context;
	PGPKeyDBObj      	*signkey = NULL;
	PGPUInt32			 revnum = 0;
	PGPSigSpec *		 sigspec;
	PGPError			 error = kPGPError_NoErr;
	
	if( pgpObjectType( key ) == RINGTYPE_SUBKEY )
		return pgpRevokeSubKey_internal( key, passphrase, passphraseLength,
							hashedPhrase, cacheTimeOut, cacheGlobal );
	
	keys =	PGPPeekKeyDBObjKeyDB( key );
	context =	PGPPeekKeyDBContext( keys );

	if ( IsPGPError( pgpKeyDeadCheck(key) ) )
	   return kPGPError_NoErr;	/* no need */
	
	for ( ; ; ) {
		signkey = key;
		/* See if we have an authorized revocation signature */
		if (!pgpKeyIsSec (key)) {
			PGPByte revclass;
			signkey = pgpKeyRevocationKey (key, revnum++, NULL, NULL,
											  &revclass, NULL, &error);
			if( IsPGPError( error ) ) {
				if( error == kPGPError_ItemNotFound )
					error = kPGPError_NoErr;
				break;
			}
			if( IsNull( signkey ) )
				continue;
			if (!(revclass & 0x80))
				continue;
			if (!pgpKeyIsSec (signkey))
				continue;
		}
		error = sCreateSigSpec( context, signkey, PGP_SIGTYPE_KEY_REVOKE,
								passphrase, passphraseLength, hashedPhrase,
								cacheTimeOut, cacheGlobal, &sigspec );
		if( IsntPGPError( error ) )
			error = sCertifyObject( sigspec, key );

		/* Retry if bad passphrase and we are an authorized revoker */
		if (error != kPGPError_BadPassphrase || signkey == key)
			break;
	}

	return error;
}
 

static const PGPOptionType revkeyOptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase
};

	PGPError
pgpRevokeKeyInternal(
	PGPKeyDBObjRef			key,
	PGPOptionListRef	optionList )
{
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;

	pgpa(pgpaPGPKeyValid(key));
	PGPValidateKey( key );

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						revkeyOptionSet, elemsof( revkeyOptionSet ) ) ) )
		return err;

	/* Pick up optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_Passphrase, FALSE,
						"%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( pgpFrontEndKey( key ) )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( key );

		err = pgpRevokeKey_back( PGPPeekKeyDBContext(keydb),
								 pgpKeyDBObjID(key), passphrase,
								 passphraseLength, hashedPhrase, cacheTimeOut,
								 cacheGlobal, &newobjs, &newobjslen);
		if( IsPGPError( err ) )
			return err;
		pgpKeyDBObjRefresh( key, FALSE );
		err = pgpAddFromKeyArray( keydb, NULL, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
	} else {
		err = pgpRevokeKey_internal( key, passphrase, passphraseLength,
									 hashedPhrase, cacheTimeOut, cacheGlobal );
	}

	/* Calculate trust changes as a result */
	if( err == kPGPError_NoErr )
		(void)PGPCalculateTrust( PGPPeekKeyDBObjKeyDB(key)->rootSet, NULL );

error:
	return err;
}


static const PGPOptionType keyentOptionSet[] = {
	kPGPOptionType_KeyGenParams,
	kPGPOptionType_KeyGenFast,
	kPGPOptionType_KeyFlags,
	kPGPOptionType_KeyGenUseExistingEntropy,
	kPGPOptionType_KeyGenOnToken
};

/* Return the amount of entropy needed to create a key of the specified
   type and size.  The application must call pgpRandpoolEntropy() itself
   until it has accumulated this much. */

	PGPUInt32
pgpKeyEntropyNeededInternal(
	PGPContextRef	context,
	PGPOptionListRef	optionList
	)
{
	PGPEnv				*pgpEnv;
	PGPUInt32			fastgen;
	PGPBoolean			fastgenop;
	PGPUInt32			noentropy = FALSE;
	PGPUInt32			pkalg;
	PGPUInt32			bits;
	PGPBoolean			keyflagsop;
	PGPUInt32			keyflags;
	PGPBoolean			v3;
	PGPBoolean			useToken;
	PGPUInt32			tokenID;
	void *				tok = NULL;
	PGPError			err = kPGPError_NoErr;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						keyentOptionSet, elemsof( keyentOptionSet ) ) ) )
		return err;

	/* If generating with existing entropy, we don't need any amount */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenUseExistingEntropy, FALSE,
						"%d", &noentropy ) ) )
		goto error;
	if (noentropy)
		return 0;
	
	pgpEnv = pgpContextGetEnvironment( context );
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenParams, TRUE,
						"%d%d", &pkalg, &bits ) ) )
		goto error;
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenOnToken, FALSE,
						"%b%d", &useToken, &tokenID ) ) )
		goto error;
	if( useToken )
		tok = pgpTokenGetIndexed( tokenID );

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenFast, FALSE,
						"%b%d", &fastgenop, &fastgen ) ) )
		goto error;
	if( !fastgenop ) {
		fastgen = pgpenvGetInt (pgpEnv, PGPENV_FASTKEYGEN, NULL, NULL);
	}

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyFlags, FALSE,
						"%b%d", &keyflagsop, &keyflags ) ) )
		goto error;

	v3 = (pkalg == kPGPPublicKeyAlgorithm_RSA) && !keyflagsop;
		
	return pgpSecKeyEntropy (pgpPkalgByNumber ((PGPByte)pkalg), bits,
							(PGPBoolean)fastgen, v3, tok);

	/* Should not have an error unless bad parameters */
error:
	pgpAssert(0);
	return ~(PGPUInt32)0;
}


/* Internal function for passphraseIsValid */
	PGPError
pgpPassphraseIsValid_internal(
	PGPKeyDBObjRef	key,
	const char *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPBoolean *	isValid)
{
	PGPSecKey *		seckey;
	PGPKeyDBObjRef	child;
	
	PGPError		err = kPGPError_NoErr;

	PGPValidateKey( key );
	if( IsntNull( passphrase ) )
		PGPValidateParam( passphrase );
	PGPValidateParam( isValid );

	/* Default return value */
	*isValid = FALSE;

	seckey = pgpSecSecKey (key, 0);

	/* If not a secret key, just return */
	if( !seckey )
		return err;

	err = pgpSecKeyUnlockWithCache( seckey, (PGPByte const *)passphrase,
									passphraseLength, hashedPhrase,
									cacheTimeOut, cacheGlobal );
	pgpSecKeyDestroy( seckey );
	*isValid = IsntPGPError( err );

	/* If requesting caching, also try it with all child keys */
	if( cacheTimeOut > 0 && OBJISTOPKEY( key ) && *isValid )
	{
		for( child = key->down; IsntNull( child ); child = child->next )
		{
			if( !OBJISSUBKEY( child ) )
				continue;

			seckey = pgpSecSecKey (child, 0);
			if( !seckey )
				continue;

			pgpSecKeyUnlockWithCache( seckey, (PGPByte const *)passphrase,
									  passphraseLength, hashedPhrase,
									  cacheTimeOut, cacheGlobal );
			pgpSecKeyDestroy( seckey );
		}
	}

	return kPGPError_NoErr;
}

	

static const PGPOptionType passphraseisvalidOptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase
};

	PGPBoolean
pgpPassphraseIsValidInternal(
	PGPKeyDBObjRef			key,
	PGPOptionListRef	optionList
	)
{
	char *				passphrase = NULL;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPBoolean			rslt;
	PGPError			err = kPGPError_NoErr;

	pgpa(pgpaPGPKeyValid(key));
	if ( ! pgpKeyIsValid( key ) )
		return( FALSE );
	
	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
								passphraseisvalidOptionSet,
								elemsof( passphraseisvalidOptionSet ) ) ) )
		return FALSE;

	/* Pick up optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		return FALSE;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			return FALSE;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		return FALSE;

	if( pgpFrontEndKey( key ) )
	{
		if( IsPGPError( pgpPassphraseIsValid_back( PGPPeekKeyDBObjContext(key),
												   pgpKeyDBObjID(key),
												   passphrase,passphraseLength,
												   hashedPhrase, cacheTimeOut,
												   cacheGlobal, &rslt ) ) )
			return FALSE;
	} else {
		if( IsPGPError( pgpPassphraseIsValid_internal( key, passphrase,
													   passphraseLength,
													   hashedPhrase,
													   cacheTimeOut,
													   cacheGlobal, &rslt ) ) )
			return FALSE;
	}

	return rslt;
}



/*____________________________________________________________________________
	Key Generation
____________________________________________________________________________*/


/*
 * Callback impedence matching, convert from internal state to callback
 * state.
 */
typedef struct PGPKeyGenProgressState {
	PGPContextRef			context;
	PGPEventHandlerProcPtr	progress;
	PGPUserValue			userValue;
} PGPKeyGenProgressState;
	
static int								/* Return < 0 to abort run */
genProgress(
	void *arg,
	int c
	)
{
	PGPKeyGenProgressState	*s = (PGPKeyGenProgressState *)arg;
	PGPError				err = kPGPError_NoErr;
	PGPOptionListRef		newOptionList = NULL;

	if (IsntNull (s->progress)) {
		err = pgpEventKeyGen (s->context, &newOptionList,
							 s->progress, s->userValue, (PGPUInt32)c);
		if (IsntNull (newOptionList))
			pgpFreeOptionList (newOptionList);
	}
	return err;
}



/*  Common code for generating master keys and subkeys. *masterkey
    is NULL when generating a master key, and is used to return
	the master PGPKeyDBObj.  If *masterkey contains a value,
	a subkey is to be generated associated with the PGPKeyDBObj.
	
    Depending on the circumstances, this may be called either on the
	front end or the back end.  Presently, smartcard keys are done on
	the back end and other keys on the front end.
*/

	PGPError
pgpDoGenerateKey_internal (
	PGPKeyDB *		keyDB,
	PGPKeyDBObj *	masterkey,
	PGPByte			pkalg,
	unsigned		bits,
	PGPTime			creationDate,
	PGPUInt16		expirationDays,
	char const *	name,
	int				name_len, 
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		passphraseIsKey,
	char const *	masterpass, 
	PGPSize			masterpassLength,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPEventHandlerProcPtr progress,
	PGPUserValue	userValue,
	PGPBoolean		fastgen,
	PGPBoolean		checkentropy,
	PGPBoolean		useToken,
	PGPUInt32		tokenID,
	PGPKeySet const *adkset,
	PGPByte			adkclass,
	PGPKeySet const *rakset,
	PGPByte			rakclass,
	PGPCipherAlgorithm const * prefalg,
	PGPSize			prefalgLength,
	PGPByte const *	prefkeyserv,
	PGPSize			prefkeyservLength,
	PGPUInt32		keyflags,
	PGPBoolean		fkeyflags,
	PGPUInt32		keyservprefs,
	PGPBoolean		fkeyservprefs,
	PGPKeyDBObj **	newkey)
{
	PGPKeyDBObj		    	*newobj = NULL;
	PGPError	          	error = kPGPError_NoErr;
	PGPSecKey				*seckey = NULL, *masterseckey = NULL;
	PGPKeySpec				*keyspec = NULL;
	long             		entropy_needed, entropy_available;
	PGPBoolean              genMaster = (masterkey == NULL);
	PGPEnv					*pgpEnv;
	PGPEnv					*pgpEnvCopy;
	PGPRandomContext		*pgpRng;
	PGPKeyGenProgressState	progressState;
	PGPContextRef			context	= PGPPeekKeyDBContext( keyDB );
	PGPUInt32				i;
	const PGPPkAlg *		algInfo;
	void					*tok = NULL;
	int						v3;
	
	if( useToken )
		tok = pgpTokenGetIndexed( tokenID );

	v3 = (pkalg == kPGPPublicKeyAlgorithm_RSA) && !fkeyflags;
		
	algInfo = pgpPkalgByNumber( pkalg );
	if( IsntNull( algInfo ) )
	{
		if( ( pgpKeyAlgUse( algInfo ) & PGP_PKUSE_SIGN ) == 0 &&
			genMaster )
		{
			pgpDebugMsg( "Invalid master key algorithm" );
			error = kPGPError_BadParams;
		}
		else if( ( pgpKeyAlgUse( algInfo ) & PGP_PKUSE_ENCRYPT ) == 0 &&
				! genMaster )
		{
			pgpDebugMsg( "Invalid subkey algorithm" );
			error = kPGPError_BadParams;
		}
	}
	else
	{
		pgpDebugMsg( "Invaid public key algorithm" );
		error = kPGPError_BadParams;
	}
	
	if( IsPGPError( error ) )
		goto cleanup;
		
	/* Grab system state from RNG for highest quality seeding */
	{ PGPRandomContext rc;
	  pgpInitGlobalRandomPoolContext( &rc );
	  pgpRandomCollectOsData( &rc, FALSE );
	}

	pgpEnv = pgpContextGetEnvironment( keyDB->context );
	if( checkentropy )
	{
		/* Check we have sufficient random bits to generate the keypair */
		entropy_needed = pgpSecKeyEntropy (algInfo, bits, fastgen, v3, tok);
		entropy_available = 8*PGPContextReserveRandomBytes ( context, 0 );
		if (entropy_needed > entropy_available)
		{
			error = kPGPError_OutOfEntropy;
			goto cleanup;
		}
	}
	
	/* Generate the secret key */
	progressState.progress = progress;
	progressState.userValue = userValue;
	progressState.context = keyDB->context;
	pgpRng = pgpContextGetX9_17RandomContext( keyDB->context );
	seckey = pgpSecKeyGenerate( context, algInfo, bits, fastgen, v3, pgpRng, 
				genProgress, &progressState, tok, (PGPByte *) passphrase, 
				passphraseLength, genMaster, &error);
	if (error)  {
		if( masterkey && IsntNull( tok ) )  {
			PGPKeyDBObjRef subkey = pgpKeySubkey(masterkey, NULL);

			/* If we have no other subkey, remove master key from the token */
			if( IsNull( subkey ) )  {
				PGPKeyID keyID;
				pgpKeyID8 (masterkey, NULL, &keyID);
				pgpDeleteKeyOnToken( context, &keyID, tokenID );
			}
		}

		goto cleanup;
	}
	pgpRandomStir (pgpRng);

	/* Need to lock the SecKey with the passphrase.  */
	if (!useToken && IsntNull(passphrase) && passphraseLength > 0) {
		PGPStringToKeyType s2ktype;
		if (passphraseIsKey) {
			s2ktype = kPGPStringToKey_LiteralShared;
		} else if (seckey->pkAlg <= kPGPPublicKeyAlgorithm_RSA) {
			s2ktype = kPGPStringToKey_Simple;
		} else {
			s2ktype = kPGPStringToKey_IteratedSalted;
		}

		/* Lock using key's preferred algorithm if known */

		if( IsPGPError( error = pgpenvCopy( pgpEnv, &pgpEnvCopy ) ) )
			goto cleanup;
		if( !genMaster )
		{
			/* For subkey, look at master key's preferred algs */
			PGPByte const		*prefAlgs;
			PGPSize				 prefAlgsLength;
			PGPBoolean			 hashed;
			prefAlgs = pgpKeyFindSubpacket ( masterkey,
				SIGSUB_PREFERRED_ENCRYPTION_ALGS, 0,
				&prefAlgsLength, NULL, &hashed, NULL, NULL, &error);
			for( i = 0; hashed && (i < prefAlgsLength); ++i )
			{
				PGPCipherAlgorithm lockAlg = (PGPCipherAlgorithm)prefAlgs[i];
				if( IsntNull( pgpCipherGetVTBL ( lockAlg ) ) )
				{
					pgpenvSetInt (pgpEnvCopy, PGPENV_CIPHER, lockAlg,
								  PGPENV_PRI_FORCE);
					break;
				}
			}
		} else {
			/* For master key, use prefalg parameter */
			for( i = 0; i < prefalgLength/sizeof(PGPCipherAlgorithm); ++i )
			{
				PGPCipherAlgorithm lockAlg = (PGPCipherAlgorithm)prefalg[i];
				if( IsntNull( pgpCipherGetVTBL ( lockAlg ) ) )
				{
					pgpenvSetInt (pgpEnvCopy, PGPENV_CIPHER, lockAlg,
								  PGPENV_PRI_FORCE);
					break;
				}
			}
		}

		error = (PGPError)pgpSecKeyChangeLock (seckey, pgpEnvCopy, pgpRng,
									NULL, 0, FALSE,		   
									passphrase, passphraseLength,
									s2ktype);
		pgpenvDestroy (pgpEnvCopy);
		
		if (error)
			goto cleanup;
	}

	/*  Generate the keyring objects.  Use keyspec defaults except for 
		expiration (validity) period */
	keyspec = pgpKeySpecCreate (pgpEnv);
	if (!keyspec) {
		error = kPGPError_OutOfMemory;
		goto cleanup;
	}
	
	if( creationDate != 0 )
	{
		pgpKeySpecSetCreation(keyspec, creationDate +
				(60 * 60 * pgpenvGetInt(pgpEnv, PGPENV_TZFIX,  NULL, NULL)));
	}
	
	pgpKeySpecSetValidity (keyspec, expirationDays);
	pgpKeySpecSetVersion( keyspec, v3 ? PGPVERSION_3 : PGPVERSION_4 );
	pgpKeySpecSetPkAlg( keyspec, pkalg );
	if( prefkeyservLength > 0 )
		pgpKeySpecSetPrefKeyserv( keyspec, prefkeyserv, prefkeyservLength );
	if( fkeyflags )
		pgpKeySpecSetKeyflags( keyspec, keyflags );
	if( fkeyservprefs )
		pgpKeySpecSetKeyservPrefs( keyspec, keyservprefs );

	if (genMaster) {
	   /* Generating master signing key */  

		if (prefalgLength > 0) {
			/* Convert preferred algorithm to byte array */
			PGPByte *prefalgByte;
			prefalgLength /= sizeof(PGPCipherAlgorithm);
			prefalgByte = (PGPByte *)pgpContextMemAlloc( context,
														prefalgLength, 0);
			if( IsNull( prefalgByte ) ) {
				error = kPGPError_OutOfMemory;
				goto cleanup;
			}
			for (i=0; i<prefalgLength; ++i) {
				prefalgByte[i] = (PGPByte)prefalg[i];
			}
			pgpKeySpecSetPrefAlgs( keyspec, prefalgByte, prefalgLength );
			pgpContextMemFree( context, prefalgByte );
		}

		if( pgpFrontEndKeyDB( keyDB ) )
		{
			PGPUInt32 *newobjs;
			PGPSize newobjslen;
			PGPUInt32 *adklist;
			PGPSize adklistsize;
			PGPUInt32 *raklist;
			PGPSize raklistsize;
			PGPUInt32 newkeyid;
			PGPByte *keyspecBuf;
			PGPSize keyspecSize;
			PGPByte *seckeyBuf;
			PGPSize seckeySize;

			if( IsPGPError(error = pgpKeySetFlatten( adkset,
													 &adklist, &adklistsize)))
				goto cleanup;
			if( IsPGPError(error = pgpKeySetFlatten( rakset,
													 &raklist, &raklistsize)))
			{
				PGPFreeData( adklist );
				goto cleanup;
			}
			if( IsPGPError( error = pgpKeySpecFlatten( keyspec, &keyspecBuf,
													   &keyspecSize ) ) )
			{
				PGPFreeData( raklist );
				PGPFreeData( adklist );
				goto cleanup;
			}
			seckeySize = pgpSecKeyBufferLength( seckey );
			if( IsNull( seckeyBuf = pgpContextMemAlloc( context,
														seckeySize, 0 ) ) )
			{
				error = kPGPError_OutOfMemory;
				PGPFreeData( keyspecBuf );
				PGPFreeData( adklist );
				PGPFreeData( adklist );
				goto cleanup;
			}
			pgpSecKeyToBuffer( seckey, seckeyBuf );

			error = pgpCreateKeypair_back( context, keyDB->id,
							seckeyBuf, seckeySize, keyspecBuf, keyspecSize,
							name, name_len,
							adklist, adklistsize, adkclass,
							raklist, raklistsize, rakclass,
							passphrase, passphraseLength,
							passphraseIsKey, cacheTimeOut, cacheGlobal,
							&newobjs, &newobjslen, &newkeyid);
			if( IsntPGPError( error ) )
			{
				error = pgpAddFromKeyArray( keyDB, NULL, newobjs, 1, TRUE );
				PGPFreeData( newobjs );
			}
			if( IsntPGPError( error ) )
			{
				error = PGPFindNode( keyDB->idToObj, newkeyid,
											   (PGPUserValue *)&newobj );
			}
		} else {
			newobj = pgpCreateKeypair (pgpEnv, keyDB, seckey, keyspec,
							name, name_len,
							adkset, adkclass, rakset, rakclass,
							(PGPByte *) passphrase, passphraseLength,
							passphraseIsKey, &error);
		}
	}
	else {
	   /* Generating encryption subkey.  Get the master seckey and 
		  unlock it */
	   masterseckey = pgpSecSecKey (masterkey, PGP_PKUSE_SIGN);
		if (!masterseckey) {
			error = pgpKeyDBError(PGPPeekKeyDBObjKeyDB(masterkey));
			goto cleanup;
		}
		if (pgpSecKeyIslocked (masterseckey)) {
		   if (IsNull( masterpass )) {
			   error = kPGPError_BadPassphrase;
			   goto cleanup;
			}
			error = (PGPError)pgpSecKeyUnlock (masterseckey,
										masterpass, masterpassLength, FALSE);
			if (error != 1) {
			   if (error == 0) 
				   error = kPGPError_BadPassphrase;
				goto cleanup;
			}
		}
		if( pgpFrontEndKeyDB( keyDB ) )
		{
			PGPUInt32 *newobjs;
			PGPSize newobjslen;
			PGPUInt32 newkeyid;
			PGPByte *keyspecBuf;
			PGPSize keyspecSize;
			PGPByte *seckeyBuf;
			PGPSize seckeySize;

			if( IsPGPError( error = pgpKeySpecFlatten( keyspec, &keyspecBuf,
													   &keyspecSize ) ) )
			{
				goto cleanup;
			}
			seckeySize = pgpSecKeyBufferLength( seckey );
			if( IsNull( seckeyBuf = pgpContextMemAlloc( context,
														seckeySize, 0 ) ) )
			{
				PGPFreeData( keyspecBuf );
				error = kPGPError_OutOfMemory;
				goto cleanup;
			}
			pgpSecKeyToBuffer( seckey, seckeyBuf );

			error = pgpCreateSubkeypair_back( context,pgpKeyDBObjID(masterkey),
									seckeyBuf, seckeySize,
									keyspecBuf, keyspecSize,
									passphrase, passphraseLength,
									passphraseIsKey, cacheTimeOut, cacheGlobal,
									&newobjs, &newobjslen, &newkeyid);

			if( IsntPGPError( error ) )
			{
				error = pgpAddFromKeyArray( keyDB, masterkey,
											newobjs, 1, TRUE );
				PGPFreeData( newobjs );
			}
			if( IsntPGPError( error ) )
			{
				error = PGPFindNode( keyDB->idToObj, newkeyid,
											   (PGPUserValue *)&newobj );
			}
		} else {
			newobj = pgpCreateSubkeypair (masterkey, pgpEnv, seckey,
									 keyspec, (PGPByte *) passphrase,
									 passphraseLength, passphraseIsKey,
									 &error);
		}
	}
	if( IsntPGPError(error) && IsntNull( tok ) && (genMaster==v3) )
	{
		/* Token gets locked when we destroy seckey used for signing */
		pgpSecKeyUnlock(seckey, passphrase, passphraseLength, passphraseIsKey);
		error = pgpTokenCopyPubKeyToToken( context,
						(genMaster?newobj:masterkey), tokenID, tok );
		if( IsPGPError( error ) )
		{
			/* Delete just-generated key from token if we couldn't finish */
			PGPKeyID keyID;
			PGPBoolean hasOtherSubkey = FALSE;
			
			if( !genMaster )
				hasOtherSubkey = IsntNull( pgpKeySubkey(masterkey, NULL) );

			if( ! hasOtherSubkey )  {
				pgpKeyID8 ((genMaster?newobj:masterkey), NULL, &keyID);
				pgpDeleteKeyOnToken( context, &keyID, tokenID );
			}

			if( !genMaster )
			{
				pgpKeyID8 (newobj, NULL, &keyID);
				pgpDeleteKeyOnToken( context, &keyID, tokenID );

				/* Finally, put PGP public object on the token -- 
				   it is still a usable key */
				if( hasOtherSubkey && error==kPGPError_SmartCardOutOfMemory )  {
					PGPError err2;
					
					pgpKeyDBRemoveObject( PGPPeekKeyDBObjKeyDB(masterkey), newobj );
					err2 = pgpTokenCopyPubKeyToToken( 
									context,masterkey,tokenID, tok );
					/* We should have enough space on the token for this */
				}
			}
		}
	}

	pgpRandomStir (pgpRng);		/* this helps us count randomness in pool */
	if (error)
		goto cleanup;

	*newkey = newobj;

	pgpKeyDBChanged( keyDB, TRUE );

cleanup:
	if (seckey)
		pgpSecKeyDestroy (seckey);
	if (masterseckey)
	   pgpSecKeyDestroy (masterseckey);
	if (keyspec)
		pgpKeySpecDestroy (keyspec);
	return error;
}


static const PGPOptionType keygenOptionSet[] = {
	kPGPOptionType_KeyDBRef,
	kPGPOptionType_KeyGenParams,
	kPGPOptionType_KeyGenName,
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase,
	kPGPOptionType_Expiration,
	kPGPOptionType_CreationDate,
	kPGPOptionType_EventHandler,
	kPGPOptionType_PreferredAlgorithms,
	kPGPOptionType_PreferredKeyServer,
	kPGPOptionType_KeyServerPreferences,
	kPGPOptionType_KeyFlags,
	kPGPOptionType_AdditionalRecipientRequestKeySet,
	kPGPOptionType_RevocationKeySet,
	kPGPOptionType_KeyGenFast,
	kPGPOptionType_KeyGenUseExistingEntropy,
	kPGPOptionType_KeyGenOnToken
};

	PGPError
pgpGenerateKeyInternal(
	PGPKeyDBObjRef		*key,
	PGPOptionListRef	optionList
	)
{
	PGPContextRef		context;
	PGPKeyDBRef			keydb;
	PGPUInt32			pkalg;
	PGPUInt32			bits;
	PGPUInt32			expiration;
	PGPTime				creationDate;
	PGPByte				*name;
	PGPUInt32			nameLength;
	PGPByte				*passphrase;
	PGPUInt32			passphraseLength;
	PGPBoolean			passphraseIsKey = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPKeySetRef		adkset;
	PGPUInt32			adkclass;
	PGPKeySetRef		rakset = NULL;
	PGPUInt32			rakclass = 0;
	PGPEventHandlerProcPtr progress;
	PGPUserValue		userValue;
	PGPKeyDBObjRef			newkey;
	PGPCipherAlgorithm	*prefalg;
	PGPSize				prefalgLength;
	PGPByte				*prefkeyserv;
	PGPSize				prefkeyservLength;
	PGPEnv				*pgpEnv;
	PGPBoolean			fastgenop;
	PGPUInt32			fastgen;
	PGPBoolean			keyflagsop;
	PGPUInt32			keyflags;
	PGPBoolean			keyservprefsop;
	PGPUInt32			keyservprefs;
	PGPUInt32			noentropy = FALSE;
	PGPBoolean			useToken;
	PGPUInt32			tokenID;
	PGPError			err;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						keygenOptionSet, elemsof( keygenOptionSet ) ) ) )
		return err;

	if( IsNull( key ) )
		return kPGPError_BadParams;

	/* First pick up mandatory options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyDBRef, TRUE,
						"%p", &keydb ) ) )
		goto error;
	
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenParams, TRUE,
						"%d%d", &pkalg, &bits ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenName, TRUE,
						"%p%l", &name, &nameLength ) ) )
		goto error;

	context =	PGPPeekKeyDBContext( keydb );
	pgpEnv = pgpContextGetEnvironment( context );

	/* Now get optional parameters */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_Passphrase, FALSE,
						"%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
		if( IsntNull( passphrase ) )
			passphraseIsKey = TRUE;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CreationDate, FALSE,
						"%T", &creationDate ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_Expiration, FALSE,
						"%d", &expiration ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_EventHandler, FALSE,
						"%p%p", &progress, &userValue ) ) )
		goto error;
	
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_PreferredAlgorithms, FALSE,
						"%p%l", &prefalg, &prefalgLength ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_PreferredKeyServer, FALSE,
						"%p%l", &prefkeyserv, &prefkeyservLength ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyFlags, FALSE,
						"%b%d", &keyflagsop, &keyflags ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyServerPreferences, FALSE,
						"%b%d", &keyservprefsop, &keyservprefs ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_AdditionalRecipientRequestKeySet,
						FALSE, "%p%d", &adkset, &adkclass ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_RevocationKeySet, FALSE,
						"%p%d", &rakset, &rakclass ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenOnToken, FALSE,
						"%b%d", &useToken, &tokenID ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenFast, FALSE,
						"%b%d", &fastgenop, &fastgen ) ) )
		goto error;
	if( !fastgenop ) {
		fastgen = pgpenvGetInt (pgpEnv, PGPENV_FASTKEYGEN, NULL, NULL);
	}

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenUseExistingEntropy, FALSE,
						"%d", &noentropy ) ) )
		goto error;
	
	*key = NULL;
	newkey = NULL;		/* Necessary to flag masterkey vs subkey */

	/* Generate in back end if we are using tokens, else front end OK */
	if( pgpFrontEndKeyDB( keydb ) && useToken )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPUInt32 *adklist;
		PGPSize adklistsize;
		PGPUInt32 *raklist;
		PGPSize raklistsize;
		PGPUInt32 newkeyid;
#if PGP_WIN32
		PGPKeyDBRef olddb = NULL;
#endif	/* PGP_WIN32 */

		if( IsPGPError(err = pgpKeySetFlatten( adkset, &adklist,&adklistsize)))
			goto error;
		if( IsPGPError(err = pgpKeySetFlatten( rakset, &raklist,&raklistsize)))
		{
			PGPFreeData( adklist );
			goto error;
		}

#if PGP_WIN32
		if( IsntNull( progress ) )
		{
			/* Create temp keydb for insertion, copy to keydb */
			olddb = keydb;
			PGPNewKeyDB( context, &keydb );
		}
#endif	/* PGP_WIN32 */

		err = pgpDoGenerateKey_back (context, keydb->id, 0,
					(PGPByte)pkalg, bits, creationDate,
					(PGPUInt16)expiration, (char *) name, nameLength,
					(char const *)passphrase, passphraseLength,
					passphraseIsKey, NULL, 0, cacheTimeOut, cacheGlobal,
					progress, userValue, (PGPBoolean)fastgen,
					(PGPBoolean)!noentropy, useToken, tokenID,
					adklist, adklistsize, (PGPByte)adkclass,
					raklist, raklistsize, (PGPByte)rakclass,
					prefalg, prefalgLength, prefkeyserv, prefkeyservLength,
					keyflags, keyflagsop, keyservprefs, keyservprefsop,
					&newobjs, &newobjslen, &newkeyid );
		if( IsntPGPError( err ) )
		{
			err = pgpAddFromKeyArray( keydb, NULL, newobjs, 1, TRUE );
			PGPFreeData( newobjs );
		}
		if( IsntPGPError( err ) )
		{
			err = PGPFindNode( keydb->idToObj, newkeyid,
										   (PGPUserValue *)&newkey );
		}
#if PGP_WIN32
		if( IsntPGPError( err )  &&  IsntNull( progress ) )
		{
			/* If interrupted during keygen, don't save key */
			PGPOptionListRef newOptionList = NULL;
			err = pgpEventKeyGen (context, &newOptionList,
								  progress, userValue, (PGPUInt32)'.');
			if (IsntNull (newOptionList))
			  pgpFreeOptionList (newOptionList);
			if( IsntPGPError( err ) )
			{
				PGPCopyKeys( PGPPeekKeyDBRootKeySet( keydb ), olddb, NULL );
				pgpKeyDBAddObject( olddb, newkey, &newkey );
			}
		}
		if( IsntNull( progress ) )
		{
			PGPFreeKeyDB( keydb );
			keydb = olddb;
		}
#endif	/* PGP_WIN32 */
	} else {
		err = pgpDoGenerateKey_internal( keydb, NULL, (PGPByte)pkalg,
								bits, creationDate, (PGPUInt16)expiration,
								(const char *)name, nameLength,
								(const char *)passphrase, passphraseLength,
								passphraseIsKey, NULL, 0, cacheTimeOut,
								cacheGlobal, progress, userValue,
								(PGPBoolean)fastgen, (PGPBoolean)!noentropy,
								useToken, tokenID, adkset, (PGPByte)adkclass,
								rakset, (PGPByte)rakclass,
								prefalg, prefalgLength, prefkeyserv,
								prefkeyservLength, keyflags, keyflagsop,
								keyservprefs, keyservprefsop, &newkey );
	}
	
	if( IsntPGPError( err ) )
	    *key = newkey;

error:
	return err;
}

static const PGPOptionType subkeygenOptionSet[] = {
	 kPGPOptionType_KeyGenMasterKey,
	 kPGPOptionType_KeyGenParams,
	 kPGPOptionType_Passphrase,
	 kPGPOptionType_Passkey,
	 kPGPOptionType_CachePassphrase,
	 kPGPOptionType_Expiration,
	 kPGPOptionType_CreationDate,
	 kPGPOptionType_EventHandler,
	 kPGPOptionType_KeyFlags,
	 kPGPOptionType_KeyGenFast,
	 kPGPOptionType_KeyGenUseExistingEntropy,
	 kPGPOptionType_KeyGenOnToken
};

	PGPError
pgpGenerateSubKeyInternal(
	PGPKeyDBObjRef		*subkey,
	PGPOptionListRef	optionList
	)
{
	PGPKeyDB		   *keydb;
	PGPContextRef		context;
	PGPUInt32			pkalg;
	PGPUInt32			bits;
	PGPTime				creationDate;
	PGPUInt32			expiration;
	PGPByte				*passphrase;
	PGPUInt32			passphraseLength;
	PGPBoolean			passphraseIsKey = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPEventHandlerProcPtr progress;
	PGPUserValue		userValue;
	PGPKeyDBObjRef		masterkey;
	PGPKeyDBObjRef		newsubkey;
	PGPEnv				*pgpEnv;
	PGPBoolean			fastgenop;
	PGPUInt32			fastgen;
	PGPBoolean			keyflagsop;
	PGPUInt32			keyflags;
	PGPUInt32			noentropy = FALSE;
	PGPBoolean			useToken;
	PGPUInt32			tokenID;
	PGPError			err;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
					   subkeygenOptionSet, elemsof( subkeygenOptionSet ) ) ) )
		return err;

	if( IsNull( subkey ) )
		return kPGPError_BadParams;

	/* First pick up mandatory options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_KeyGenMasterKey, TRUE,
						 "%p", &masterkey ) ) )
		goto error;
	
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_KeyGenParams, TRUE,
						 "%d%d", &pkalg, &bits ) ) )
		goto error;


	keydb = PGPPeekKeyDBObjKeyDB( masterkey );
	context = PGPPeekKeyDBContext( keydb );
	pgpEnv = pgpContextGetEnvironment( context );

	/* Now get optional parameters */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
		if( IsntNull( passphrase ) )
			passphraseIsKey = TRUE;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_CreationDate, FALSE,
						 "%T", &creationDate ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Expiration, FALSE,
						 "%d", &expiration ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_EventHandler, FALSE,
						 "%p%p", &progress, &userValue ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyFlags, FALSE,
						"%b%d", &keyflagsop, &keyflags ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenOnToken, FALSE,
						"%b%d", &useToken, &tokenID ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_KeyGenFast, FALSE,
						 "%b%d", &fastgenop, &fastgen ) ) )
		goto error;
	if( !fastgenop ) {
		fastgen = pgpenvGetInt (pgpEnv, PGPENV_FASTKEYGEN, NULL, NULL);
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyGenUseExistingEntropy, FALSE,
						"%d", &noentropy ) ) )
		goto error;
	
	err	= pgpKeyDeadCheck(masterkey);
	if ( IsPGPError( err ) )
	    return err;

	*subkey = NULL;
	if( pgpFrontEndKeyDB( keydb ) && useToken )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPUInt32 newkeyid;
#if PGP_WIN32
		PGPKeyDBRef olddb = NULL;

		/* Win32 needs special callback handling */
		if( IsntNull( progress ) )
		{
			/* Create temp keydb for insertion, copy to keydb */
			olddb = keydb;
			PGPNewKeyDB( context, &keydb );
			PGPCopyKeyDBObj( masterkey, keydb, &masterkey );
		}
#endif	/* PGP_WIN32 */

		err = pgpDoGenerateKey_back (context, keydb->id,
					pgpKeyDBObjID(masterkey), (PGPByte)pkalg, bits,
					creationDate, (PGPUInt16)expiration, NULL, 0,
					(char const *)passphrase, passphraseLength,
					passphraseIsKey, (char const *)passphrase,
					passphraseLength, cacheTimeOut, cacheGlobal,
					progress, userValue, (PGPBoolean)fastgen,
					(PGPBoolean)!noentropy, useToken, tokenID, NULL, 0,
					(PGPByte)0, NULL, 0, (PGPByte)0, NULL, 0, NULL, 0,
					keyflags, keyflagsop, 0, (PGPBoolean)FALSE,
					&newobjs, &newobjslen, &newkeyid);
		if( IsntPGPError( err ) )
		{
			err = pgpAddFromKeyArray( keydb, masterkey, newobjs, 1, TRUE );
			PGPFreeData( newobjs );
		}
		if( IsntPGPError( err ) )
		{
			err = PGPFindNode( keydb->idToObj, newkeyid,
										   (PGPUserValue *)&newsubkey );
		}
#if PGP_WIN32
		if( IsntPGPError( err ) && IsntNull( progress ) )
		{
			/* One last chance for an interruption */
			PGPOptionListRef newOptionList = NULL;
			err = pgpEventKeyGen (context, &newOptionList,
								  progress, userValue, (PGPUInt32)'.');
			if (IsntNull (newOptionList))
				pgpFreeOptionList (newOptionList);
			if( IsntPGPError( err ) )
			{
				PGPCopyKeys( PGPPeekKeyDBRootKeySet( keydb ), olddb, NULL );
				pgpKeyDBAddObject (olddb, newsubkey, &newsubkey );
			}
		}
		if( IsntNull( progress ) )
		{
			PGPFreeKeyDB( keydb );
			keydb = olddb;
		}
#endif	/* PGP_WIN32 */
	} else {
		err = pgpDoGenerateKey_internal (keydb, masterkey,
								   (PGPByte)pkalg, bits, creationDate,
								   (PGPUInt16)expiration,
								   NULL, 0, (char const *)passphrase,
								   passphraseLength, passphraseIsKey,
								   (char const *)passphrase, passphraseLength,
								   cacheTimeOut, cacheGlobal,
								   progress, userValue,
								   (PGPBoolean)fastgen, (PGPBoolean)!noentropy,
								   useToken, tokenID, NULL, (PGPByte)0,
								   NULL, (PGPByte)0, NULL, 0,
								   NULL, 0, keyflags, keyflagsop, 0,
								   (PGPBoolean)FALSE, &newsubkey);
	}
	if( IsntPGPError( err ) )
	    *subkey = newsubkey;

error:
	return err;
}


/*  Handle editing key properties which are held in self signatures  */


static const PGPOptionType keyoptionOptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase,
	kPGPOptionType_RevocationKeySet,
	kPGPOptionType_PreferredAlgorithms,
	kPGPOptionType_PreferredKeyServer,
	kPGPOptionType_KeyServerPreferences,
	kPGPOptionType_KeyFlags,
#if 0
/* not yet implemented */
	kPGPOptionType_Expiration,
	kPGPOptionType_AdditionalRecipientRequestKeySet,
#endif
};


	PGPError
pgpAddKeyOptions_internal (
	PGPKeyDBObjRef		key,
	char *				passphrase,
	PGPSize				passphraseLength,
	PGPBoolean			hashedPhrase,
	PGPUInt32			cacheTimeOut,
	PGPBoolean			cacheGlobal,
	PGPKeySetRef		rakset,
	PGPUInt32			rakclass
	)
{
	PGPKeySetRef		rak1set = NULL;
	PGPSigSpec			*sigspec;
	PGPKeyIter			*rakiter;
	PGPKeyDB			*keys;
	PGPKeyDBObj			*rakkey;
	PGPContextRef		context;
	PGPError			err = kPGPError_NoErr;

	keys = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keys );

	err = PGPNewKeyIterFromKeySet( rakset, &rakiter );
	if( IsPGPError( err ) )
		goto error;

	/* Add 1 RAK key at a time in separate self signatures */
	while( IsntPGPError( PGPKeyIterNextKeyDBObj( rakiter, kPGPKeyDBObjType_Key,
					&rakkey ) ) ) {
		pgpAssert (pgpObjectType(rakkey) == RINGTYPE_KEY);
		err = PGPNewOneKeySet( rakkey, &rak1set );
		if( IsPGPError( err ) )
			goto error;

		err = sCreateSigSpec( context, key, PGP_SIGTYPE_KEY_PROPERTY,
							  passphrase, passphraseLength, hashedPhrase,
							  cacheTimeOut, cacheGlobal, &sigspec );
		if( IsntPGPError( err ) )
			err = sSigSpecSetExportability( sigspec, SIG_EXPORTABLE, 0 );
		if( IsntPGPError( err ) )
			err = sSigSpecAddRAK( sigspec, rak1set, rakclass );

		if( IsntPGPError( err ) )
			err = sCertifyObject( sigspec, key );
		if( IsPGPError( err ) ) {
			goto error;
		}
		PGPFreeKeySet (rak1set);
		rak1set = NULL;
	}
	PGPFreeKeyIter( rakiter );
	rakiter = NULL;

error:

	if( IsntNull( rak1set ) )
		PGPFreeKeySet( rak1set );
	if( IsntNull( rakiter ) )
		PGPFreeKeyIter (rakiter);

	return err;
}


	PGPError
pgpAddKeyOptionsInternal (
	PGPKeyDBObjRef			key,
	PGPOptionListRef	optionList
	)
{
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPKeySetRef		rakset = NULL;
	PGPUInt32			rakclass = 0;
	PGPError			err = kPGPError_NoErr;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						keyoptionOptionSet, elemsof( keyoptionOptionSet ) ) ) )
		goto error;

	/* Pick up passphrase options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	/* Get data to add (require revocationkeyset for now) */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_RevocationKeySet, TRUE,
						"%p%d", &rakset, &rakclass ) ) )
		goto error;
	pgpAssert( IsntNull( rakset ) );


	/*
	 * This code is temporary and will be redesigned to support a wider
	 * set of key options.
	 */

	if ( IsPGPError( err = pgpKeyDeadCheck(key) ) ) {
		goto error;
	}
	
	if( pgpFrontEndKey( key ) )
	{
		PGPUInt32 *raklist;
		PGPSize raklistsize;
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( key );

		if( IsPGPError(err = pgpKeySetFlatten( rakset, &raklist,&raklistsize)))
			goto error;
		err = pgpAddKeyOptions_back( PGPPeekKeyDBContext(keydb),
									 pgpKeyDBObjID(key), passphrase,
									 passphraseLength, hashedPhrase,
									 cacheTimeOut, cacheGlobal,
									 raklist, raklistsize, rakclass,
									 &newobjs, &newobjslen);
		if( IsPGPError( err ) )
			goto error;
		err = pgpAddFromKeyArray( keydb, NULL, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
	} else {
		err = pgpAddKeyOptions_internal( key, passphrase, passphraseLength,
										 hashedPhrase, cacheTimeOut,
										 cacheGlobal, rakset, rakclass );
		if( IsPGPError( err ) )
			goto error;
	}

	/* Calculate trust changes as a result */
	if( err == kPGPError_NoErr )
		(void)PGPCalculateTrust (PGPPeekKeyDBObjKeyDB(key)->rootSet, NULL);

error:
	return err;
}

	PGPError
pgpRemoveKeyOptionsInternal (
	PGPKeyDBObjRef			key,
	PGPOptionListRef	optionList
	)
{
	(void) key;
	(void) optionList;
	return kPGPError_FeatureNotAvailable;
}

	PGPError
pgpUpdateKeyOptions_internal (
	PGPKeyDBObjRef	   key,
	char *			   passphrase,
	PGPSize			   passphraseLength,
	PGPBoolean		   hashedPhrase,
	PGPUInt32		   cacheTimeOut,
	PGPBoolean		   cacheGlobal,
	PGPCipherAlgorithm	*prefalg,
	PGPSize				prefalgLength,
	PGPByte				*prefkeyserv,
	PGPSize				prefkeyservLength,
	PGPUInt32			keyflags,
	PGPBoolean			fkeyflags,
	PGPUInt32			keyservprefs,
	PGPBoolean			fkeyservprefs
	)
{
	PGPKeyDB		  *keys;
	PGPEnv			  *pgpEnv;
	PGPSigSpec		  *sigspec = NULL;
	int				   tzFix;
	PGPTime			   timestamp;
	PGPContextRef	   context;
	PGPByte			   *prefalgByte = NULL;
	PGPKeyDBObj		   *userid;
	PGPKeyDBObj		   *latestsig;
	PGPByte				flagbuf[4];
	PGPUInt32		    i;
	PGPError			err = kPGPError_NoErr;

	keys = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keys );
	pgpEnv = pgpContextGetEnvironment( context );
	tzFix  = pgpenvGetInt (pgpEnv, PGPENV_TZFIX, NULL, NULL);
	timestamp = pgpTimeStamp (tzFix);

	/* Parse preferred algorithms into tight buffer */
	prefalgByte = NULL;
	if (prefalgLength > 0) {
		/* Convert preferred algorithm to byte array */
		prefalgLength /= sizeof(PGPCipherAlgorithm);
		prefalgByte = (PGPByte *)pgpContextMemAlloc( context,
													prefalgLength, 0);
		if( IsNull( prefalgByte ) ) {
			err = kPGPError_OutOfMemory;
			goto error;
		}
		for (i=0; i<prefalgLength; ++i) {
			prefalgByte[i] = (PGPByte)prefalg[i];
		}
	}

	/* Verify passphrase */
	if( !pgpSecPassphraseOK( key, (PGPByte *) passphrase, passphraseLength,
							 hashedPhrase, cacheTimeOut, cacheGlobal  ) )
	{
		err = kPGPError_BadPassphrase;
		goto error;
	}

	/* Update each self-sig on each name */
	for (userid = key->down; IsntNull( userid ); userid = userid->next) {
		if (!pgpKeyDBObjIsReal( userid) )
			continue;
		if( pgpObjectType( userid ) != RINGTYPE_USERID )
			continue;
	
		latestsig = pgpLatestSigByKey( userid, key );
		if( IsntNull( latestsig ) )
		{
			/* Revoke existing cert */
			sigspec = pgpSigSpecCreate (pgpEnv, key,
										PGP_SIGTYPE_KEY_UID_REVOKE);
			pgpSigSpecSetTimestamp( sigspec, timestamp );
			err = pgpSignObject (userid, sigspec);
			pgpSigSpecDestroy (sigspec);
			sigspec = NULL;

			/* Copy existing sigspec, set new timestamp */
			sigspec = pgpSigSigSpec (latestsig, &err);
			if( IsNull( sigspec ) )
				continue;
			pgpSigSpecSetTimestamp( sigspec, timestamp+1 );

		}
		else
		{
			/* No previous sig, must create one */
			sigspec = pgpSigSpecCreate (pgpEnv, key, PGP_SIGTYPE_KEY_GENERIC);
			if( IsNull( sigspec ) )
				continue;
			pgpSigSpecSetTimestamp( sigspec, timestamp+1 );
		}

		/* Set new values */
		pgpSigSpecSetPrefAlgs (sigspec, 0, prefalgByte, prefalgLength);
		pgpSigSpecSetPrefKeyServ (sigspec, 0, prefkeyserv, prefkeyservLength);

		if( fkeyflags )
		{
			for( i=0; i<sizeof(flagbuf); ++i )
				flagbuf[i] = (PGPByte)(keyflags >> (i*8));
			pgpSigSpecSetKeyFlags( sigspec, 0, flagbuf, sizeof(flagbuf) );
		}
		if( fkeyservprefs )
		{
			for( i=0; i<sizeof(flagbuf); ++i )
				flagbuf[i] = (PGPByte)(keyservprefs >> (i*8));
			pgpSigSpecSetKeyServPrefs( sigspec, 0, flagbuf, sizeof(flagbuf) );
		}

		/* Issue new signature */
		err = pgpSignObject (userid, sigspec);
		pgpSigSpecDestroy (sigspec);
		sigspec = NULL;

	}

error:

	if( IsntNull( sigspec ) )
		pgpSigSpecDestroy (sigspec);
	if( IsntNull( prefalgByte ) )
		PGPFreeData (prefalgByte);

	return err;
}


	PGPError
pgpUpdateKeyOptionsInternal (
	PGPKeyDBObjRef			key,
	PGPOptionListRef	optionList
	)
{
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPCipherAlgorithm	*prefalg;
	PGPSize				prefalgLength;
	PGPByte				*prefkeyserv;
	PGPSize				prefkeyservLength;
	PGPBoolean			keyflagsop;
	PGPUInt32			keyflags;
	PGPBoolean			keyservprefsop;
	PGPUInt32			keyservprefs;
	PGPError			err = kPGPError_NoErr;


	if ( IsPGPError( err = pgpKeyDeadCheck(key) ) )
		goto error;
	
	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						keyoptionOptionSet, elemsof( keyoptionOptionSet ) ) ) )
		goto error;

	/* Pick up passphrase options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	/* Get data to modify (require preferred algs  for now) */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_PreferredAlgorithms, FALSE,
						"%p%l", &prefalg, &prefalgLength ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_PreferredKeyServer, FALSE,
						"%p%l", &prefkeyserv, &prefkeyservLength ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyFlags, FALSE,
						"%b%d", &keyflagsop, &keyflags ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_KeyServerPreferences, FALSE,
						"%b%d", &keyservprefsop, &keyservprefs ) ) )
		goto error;

	if( IsNull(prefalg) && IsNull(prefkeyserv) && !keyflagsop
		&& !keyservprefsop )
		goto error;		/* Nothing to do */

	if( pgpFrontEndKey( key ) )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( key );

		err = pgpUpdateKeyOptions_back( PGPPeekKeyDBContext(keydb),
										pgpKeyDBObjID(key), passphrase,
										passphraseLength, hashedPhrase,
										cacheTimeOut, cacheGlobal,
										prefalg, prefalgLength,
										prefkeyserv, prefkeyservLength,
										keyflags, keyflagsop, keyservprefs,
										keyservprefsop, &newobjs, &newobjslen);

		if( IsPGPError( err ) )
			return err;
		err = pgpAddFromKeyArray( keydb, NULL, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
	} else {
		err = pgpUpdateKeyOptions_internal( key, passphrase, passphraseLength,
										hashedPhrase, cacheTimeOut,
										cacheGlobal, prefalg, prefalgLength,
										prefkeyserv, prefkeyservLength,
										keyflags, keyflagsop, keyservprefs,
										keyservprefsop );
		if( IsPGPError( err ) )
			return err;
	}

	/* Calculate trust changes as a result */
	if( err == kPGPError_NoErr )
		(void)PGPCalculateTrust (PGPPeekKeyDBObjKeyDB(key)->rootSet, NULL);

error:

	return err;
}





/*  Disable the key.  If key is not stored in a writeable KeySet, copy it 
	locally.  Private keys cannot be disabled. */

	PGPError
pgpDisableKey (PGPKeyDBObj *key)
{
	PGPKeyDB		   *keys = NULL;
	PGPError			error = kPGPError_NoErr;

	keys = PGPPeekKeyDBObjKeyDB( key );

	/*  Axiomatic keys cannot be disabled, but plain old private
	    keys can (because they may belong to someone else).  */
	if (pgpKeyAxiomatic (key))
		return kPGPError_BadParams;

	if (!pgpKeyDisabled (key)) {
	    pgpKeyDisable (key);
		pgpKeyDBChanged (keys, TRUE);
	}

	return error;
}


/*  Enable the key. */

	PGPError
pgpEnableKey (PGPKeyDBObj *key)
{
	PGPKeyDB		   *keys;
	
	keys = PGPPeekKeyDBObjKeyDB( key );

	if (pgpKeyDisabled (key)) {
	  	pgpKeyEnable (key);
		pgpKeyDBChanged (keys, TRUE);
	}
	return kPGPError_NoErr;
}

	PGPError
PGPSetKeyEnabled( PGPKeyDBObjRef key, PGPBoolean enable )
{
	PGPError		    error = kPGPError_NoErr;

	PGPValidateKey( key );

	pgpEnterPGPErrorFunction();

	error	= pgpKeyDeadCheck( key) ;
	if ( IsPGPError( error ) )
		return error;
		
	if( pgpFrontEndKey( key ) )
	{
		error = pgpSetKeyEnabled_back( PGPPeekKeyDBObjContext(key),
									   pgpKeyDBObjID( key ), enable );
		if( IsntPGPError( error ) )
		{
			pgpKeyDBObjRefresh( key, FALSE );
			pgpKeyDBChanged (PGPPeekKeyDBObjKeyDB(key), TRUE);
		}
	} else if( enable ) {
		error = pgpEnableKey( key );
	} else {
		error = pgpDisableKey( key );
	}

	if( IsntPGPError( error ) )
		error = PGPCalculateTrust( PGPPeekKeyDBObjKeyDB(key)->rootSet, NULL );
	return error;
}


/*  Change the passphrase.  If the new passphrase is the same as the
	old passphrase, we still unlock the key as the user may be trying to
	set the key's isAxiomatic flag.  */

	PGPError
pgpDoChangePassphrase_internal (PGPKeyDB *keyDB,
							 PGPKeyDBObj *key, PGPKeyDBObj *masterkey, 
							 const char *oldphrase, PGPSize oldphraseLength,
							 const char *newphrase, PGPSize newphraseLength,
							 PGPBoolean newPassphraseIsKey,
							 PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal)
{
	PGPError			 error = kPGPError_NoErr;
	PGPSecKey			*seckey = NULL;
	PGPEnv				*pgpEnv;
	PGPEnv				*pgpEnvCopy = NULL;
	PGPRandomContext	*pgpRng;
	PGPStringToKeyType	 s2ktype;
	PGPByte const		*prefAlgs;
	PGPSize				 prefAlgsLength;
	PGPBoolean			 hashed;
	PGPUInt32			 i;
	PGPBoolean			 locked = 0;

	if (IsntNull(oldphrase) && oldphraseLength == 0)
		oldphrase = NULL;
	if (IsntNull(newphrase) && newphraseLength == 0)
		newphrase = NULL;

	if (!pgpKeyIsSec ( key ))
	    return kPGPError_SecretKeyNotFound;

	/* Does the caller know the current passphrase? */
	pgpEnv = pgpContextGetEnvironment( keyDB->context );
	seckey = pgpSecSecKey (key, 0);
	if (!seckey)
	    return pgpKeyDBError(keyDB);
	if (pgpSecKeyIslocked (seckey)) {
		locked = 1;
	    if (!oldphrase) {
		    error = kPGPError_BadPassphrase;
			goto error;
		}
		error = (PGPError)pgpSecKeyUnlock (seckey, oldphrase, 
								 oldphraseLength, FALSE);
		if (error != 1) {
		    if (error == 0) 
			    error = kPGPError_BadPassphrase;
			goto error;
		}
	}
	
	pgpRng = pgpContextGetX9_17RandomContext( keyDB->context );

	if (newPassphraseIsKey) {
		s2ktype = kPGPStringToKey_LiteralShared;
	} else if (seckey->pkAlg <= kPGPPublicKeyAlgorithm_RSA) {
		s2ktype = kPGPStringToKey_Simple;
	} else {
		s2ktype = kPGPStringToKey_IteratedSalted;
	}

	/* Lock using key's preferred algorithm if known */
	if( IsPGPError( error = pgpenvCopy( pgpEnv, &pgpEnvCopy ) ) )
		goto error;
	prefAlgs = pgpKeyFindSubpacket (
			(IsntNull(masterkey)?masterkey:key),
			SIGSUB_PREFERRED_ENCRYPTION_ALGS, 0,
			&prefAlgsLength, NULL, &hashed, NULL, NULL, &error);
	for( i = 0; hashed && (i < prefAlgsLength); ++i ) {
		PGPCipherAlgorithm lockAlg = (PGPCipherAlgorithm)prefAlgs[i];
		if( IsntNull( pgpCipherGetVTBL ( lockAlg ) ) ) {
			pgpenvSetInt (pgpEnvCopy, PGPENV_CIPHER, lockAlg,
						  PGPENV_PRI_FORCE);
			break;
		}
	}

    error = (PGPError)pgpSecKeyChangeLock (seckey, pgpEnvCopy, pgpRng, 
								 oldphrase, oldphraseLength, FALSE,
								 newphrase, newphraseLength, s2ktype);
	if (error)
	    goto error;

	error = pgpUpdateSecKeyData( key, seckey );
	if( IsPGPError( error ) )
		goto error;

	/* Add to cache if requested */
	(void)pgpSecKeyCachePassphrase( seckey, (PGPByte *)newphrase,
									newphraseLength, newPassphraseIsKey,
									cacheTimeOut, cacheGlobal );

	pgpKeyDBChanged (keyDB, TRUE);
	pgpSecKeyDestroy (seckey); seckey = NULL;

error:
	if (seckey)
		pgpSecKeyDestroy (seckey);
	if (pgpEnvCopy)
		pgpenvDestroy (pgpEnvCopy);
	return error;
}


static const PGPOptionType changepassphraseOptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase
};

	PGPError
pgpChangePassphraseInternal(
	PGPKeyDBObjRef			key,
	PGPOptionListRef	optionList
	)
{
	PGPKeyDBRef			keydb;
	void *				oldPassphrase;
	void *				newPassphrase;
	PGPSize				oldPassphraseLength;
	PGPSize				newPassphraseLength;
	PGPBoolean			oldWasPasskey;
	PGPBoolean			newWasPasskey;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;

	pgpa(pgpaPGPKeyValid(key));
	PGPValidateKey( key );
	
	oldWasPasskey = FALSE;
	newWasPasskey = FALSE;

	keydb = PGPPeekKeyDBObjKeyDB( key );

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
									changepassphraseOptionSet,
									elemsof( changepassphraseOptionSet ) ) ) )
		goto error;

	/*
	 * Read old and new passphrases
	 */

	if( 2 != pgpSearchPassphraseOptions( optionList, &oldPassphrase,
										 &oldPassphraseLength, &oldWasPasskey,
										 &newPassphrase, &newPassphraseLength,
										 &newWasPasskey ) )
	{
		err = kPGPError_BadParams;
		goto error;
	}


	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( pgpFrontEndKey( key ) )
	{
		err = pgpDoChangePassphrase_back (PGPPeekKeyDBContext(keydb),
							 keydb->id, pgpKeyDBObjID(key), 0,
							 (char const *)oldPassphrase, oldPassphraseLength,
							 (char const *)newPassphrase, newPassphraseLength,
							 newWasPasskey, cacheTimeOut, cacheGlobal );
		if( IsntPGPError( err ) )
		{
			pgpKeyDBObjRefresh( key, FALSE );
			pgpKeyDBChanged (keydb, TRUE);
		}
	} else {
		err = pgpDoChangePassphrase_internal (keydb, key, NULL,
							 (char const *)oldPassphrase, oldPassphraseLength,
							 (char const *)newPassphrase, newPassphraseLength,
							 newWasPasskey, cacheTimeOut, cacheGlobal );
	}
error:

	return err;
}


	PGPError
pgpChangeSubKeyPassphraseInternal(
	PGPKeyDBObjRef		subkey,
	PGPOptionListRef	optionList
	)
{
	PGPKeyDBRef			keydb;
	PGPKeyDBObjRef		key;
	PGPOption			oldOp, newOp;
	void *				oldPassphrase;
	void *				newPassphrase;
	PGPSize				oldPassphraseLength;
	PGPSize				newPassphraseLength;
	PGPBoolean			oldWasPasskey;
	PGPBoolean			newWasPasskey;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;

	PGPValidateSubKey( subkey );

	key = subkey->up;
	PGPValidateKey( key );
	
	oldWasPasskey = FALSE;
	newWasPasskey = FALSE;

	keydb = PGPPeekKeyDBObjKeyDB( key );

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
									changepassphraseOptionSet,
									elemsof( changepassphraseOptionSet ) ) ) )
		goto error;

	/*
	 * Can't use our regular parsing functions because we are allowing
	 * the same option to be used more than once.
	 */

	/*
	 * Can't use our regular parsing functions because we are allowing
	 * the same option to be used more than once.
	 */

	/* Pick up old passphrase */
	if( IsPGPError( err = pgpGetIndexedOption( optionList,
					  0, TRUE, &oldOp ) ) )
		goto error;
	
	oldWasPasskey = oldOp.type == kPGPOptionType_Passkey;

	if( !oldWasPasskey && oldOp.type != kPGPOptionType_Passphrase )
	{
		err = kPGPError_BadParams;
		goto error;
	}
		
	/* Pick up new passphrase */
	if( IsPGPError( err = pgpGetIndexedOption( optionList,
					  1, TRUE, &newOp ) ) )
		goto error;
	
	newWasPasskey = newOp.type == kPGPOptionType_Passkey;

	if( !newWasPasskey && newOp.type != kPGPOptionType_Passphrase )
	{
		err = kPGPError_BadParams;
		goto error;
	}
		
	pgpOptionPtrLength( &oldOp, &oldPassphrase, &oldPassphraseLength );
	pgpOptionPtrLength( &newOp, &newPassphrase, &newPassphraseLength );

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	CHECKREMOVED(subkey);

	if( pgpFrontEndKey( key ) )
	{
		err = pgpDoChangePassphrase_back (PGPPeekKeyDBContext(keydb),
						 keydb->id, pgpKeyDBObjID(subkey),
						 pgpKeyDBObjID(key), (char const *)oldPassphrase,
						 oldPassphraseLength, (char const *)newPassphrase,
						 newPassphraseLength, newWasPasskey, cacheTimeOut,
						 cacheGlobal );
		if( IsntPGPError( err ) )
		{
			pgpKeyDBObjRefresh( subkey, FALSE );
			pgpKeyDBChanged (keydb, TRUE);
		}
	} else {
		err = pgpDoChangePassphrase_internal (keydb, subkey, key,
						 (char const *)oldPassphrase, oldPassphraseLength,
						 (char const *)newPassphrase, newPassphraseLength,
						 newWasPasskey, cacheTimeOut, cacheGlobal );
	}

error:

	return err;
}


	PGPError
pgpRevokeSubKey_internal (
	PGPKeyDBObjRef		subkey,
	char const *		passphrase,
	PGPSize				passphraseLength,
	PGPBoolean			hashedPhrase,
	PGPUInt32			cacheTimeOut,
	PGPBoolean			cacheGlobal
	)
{
    PGPKeyDB			*keys;
	PGPKeyDBObj    		*key;
	PGPKeyDBObj			*signkey = NULL;
	PGPUInt32			 revnum;
	PGPSigSpec			*sigspec;
	PGPContextRef		 context;
	PGPError			 error = kPGPError_NoErr;
	
	CHECKREMOVED(subkey);
	keys = PGPPeekKeyDBObjKeyDB( subkey );
	context = PGPPeekKeyDBContext( keys );
	key = subkey->up;
	pgpAssert( pgpObjectType( key ) == RINGTYPE_KEY );

	if (pgpSubKeyIsDead (subkey))
	    return kPGPError_NoErr;
	
	revnum = 0;
	for ( ; ; )
	{
		signkey = key;
		/* See if we have an authorized revocation signature */
		if (!pgpKeyIsSec (key))
		{
			PGPByte revclass;
			signkey = pgpKeyRevocationKey (key, revnum++, NULL, NULL,
										   &revclass, NULL, &error);
			if( IsPGPError( error ) )
			{
				if( error == kPGPError_ItemNotFound )
					error = kPGPError_NoErr;
				break;
			}
			if( IsNull( signkey ) )
				continue;
			if (!(revclass & 0x80))
				continue;
			if (!pgpKeyIsSec (signkey))
				continue;
		}
		/*  Note special subkey revocation sigtype */
		error = sCreateSigSpec( context, signkey,
								PGP_SIGTYPE_KEY_SUBKEY_REVOKE, passphrase,
								passphraseLength, hashedPhrase, cacheTimeOut,
								cacheGlobal, &sigspec );
		if( IsntPGPError( error ) )
			error = sSigSpecSetExportability( sigspec, SIG_EXPORTABLE, 0 );

		if( IsntPGPError( error ) )
			error = sCertifyObject( sigspec, subkey );

		/* Retry if bad passphrase and we are an authorized revoker */
		if (error != kPGPError_BadPassphrase || signkey == key)
			break;
	}
			
	return error;
}


static const PGPOptionType revsubkeyOptionSet[] = {
	 kPGPOptionType_Passphrase,
	 kPGPOptionType_Passkey,
	 kPGPOptionType_Passphrase
};

	PGPError
pgpRevokeSubKeyInternal(
	PGPKeyDBObjRef		subkey,
	PGPOptionListRef	optionList
	)
{
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;

	pgpa(pgpaPGPSubKeyValid(subkey));
	PGPValidateSubKey( subkey );

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
					   revsubkeyOptionSet, elemsof( revsubkeyOptionSet ) ) ) )
		return err;

	/* Pick up optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;


	if( pgpFrontEndKey( subkey ) )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( subkey );

		/* We overload the RevokeKey function */
		err = pgpRevokeKey_back( PGPPeekKeyDBContext(keydb),
								 pgpKeyDBObjID(subkey), passphrase,
								 passphraseLength, hashedPhrase, cacheTimeOut,
								 cacheGlobal, &newobjs, &newobjslen);
		if( IsPGPError( err ) )
			return err;
		pgpKeyDBObjRefresh( subkey, FALSE );
		err = pgpAddFromKeyArray( keydb, subkey->up, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
	} else {
		err = pgpRevokeSubKey_internal( subkey, passphrase, passphraseLength,
									 hashedPhrase, cacheTimeOut, cacheGlobal );
	}

	/* Calculate trust changes as a result */
	if( err == kPGPError_NoErr )
		(void)PGPCalculateTrust( PGPPeekKeyDBObjKeyDB(subkey)->rootSet, NULL );

error:
	return err;
}


/*  Convert a passphrase to a passkeybuffer, for a given key */

static const PGPOptionType getpasskeyOptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_CachePassphrase
};

	PGPError
pgpGetPasskeyBuffer_internal (
	PGPKeyDBObjRef		key,
	char const *		passphrase,
	PGPSize				passphraseLength,
	PGPByte				*passkeyBuffer
	)
{
	PGPContextRef		context;
	PGPKeyDBRef			keydb;
	PGPEnv			   *pgpEnv;
	PGPSecKey		   *secKey;
	PGPError			err;
	
	keydb = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keydb );
	pgpEnv = pgpContextGetEnvironment( context );

	secKey = pgpSecSecKey( key, 0 );
	if( IsNull( secKey ) )
	    return pgpKeyDBError(keydb);

	err =  pgpSecKeyConvertPassphrase( secKey, pgpEnv, passphrase,
						  passphraseLength, passkeyBuffer );
	pgpSecKeyDestroy (secKey);

	return err;
}


	PGPError
pgpGetKeyPasskeyBufferInternal (
	PGPKeyDBObjRef			key,
	void			   *passkeyBuffer,
	PGPOptionListRef	optionList
	)
{
	PGPError			err;
	char const		   *passphrase;
	PGPSize				passphraseLength;
	
	/* Pick up mandatory passphrase option */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, TRUE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		return err;

	if( pgpFrontEndKey( key ) )
	{
		PGPByte *buf;
		PGPSize buflength;

		err = pgpGetPasskeyBuffer_back( PGPPeekKeyDBObjContext( key ),
										pgpKeyDBObjID(key), passphrase,
										passphraseLength, &buf, &buflength );
		if( IsPGPError( err ) )
			return err;
		pgpCopyMemory( buf, (PGPByte *)passkeyBuffer, buflength );
		PGPFreeData( buf );
		return kPGPError_NoErr;
	} else {
		return pgpGetPasskeyBuffer_internal( key, passphrase, passphraseLength,
											 (PGPByte *)passkeyBuffer );
	}
}



/*  Convert a passphrase to a passkeybuffer, for a given subkey */

	PGPError
pgpGetSubKeyPasskeyBufferInternal (
	PGPKeyDBObjRef		subkey,
	void			   *passkeyBuffer,
	PGPOptionListRef	optionList
	)
{
	PGPError			err;
	char const		   *passphrase;
	PGPSize				passphraseLength;
	
	/* Pick up mandatory passphrase option */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, TRUE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		return err;

	if( pgpFrontEndKey( subkey ) )
	{
		PGPByte *buf;
		PGPSize buflength;

		err = pgpGetPasskeyBuffer_back( PGPPeekKeyDBObjContext( subkey ),
										pgpKeyDBObjID(subkey), passphrase,
										passphraseLength, &buf, &buflength );
		if( IsPGPError( err ) )
			return err;
		pgpCopyMemory( buf, (PGPByte *)passkeyBuffer, buflength );
		PGPFreeData( buf );
		return kPGPError_NoErr;
	} else {
		return pgpGetPasskeyBuffer_internal( subkey, passphrase,
											 passphraseLength,
											 (PGPByte *)passkeyBuffer );
	}
}


/*
 *  Delete a keydb obj.
 *  If a userid, make sure we don't remove the last one.
 */
	PGPError
PGPDeleteKeyDBObj (PGPKeyDBObj *obj)
{
	PGPKeyDBRef			keys;
	PGPError			err;

	PGPValidateKeyDBObj( obj );

	pgpEnterPGPErrorFunction();

	CHECKREMOVED(obj);

	keys = PGPPeekKeyDBObjKeyDB(obj);
	err = pgpKeyDBRemoveObject( keys, obj);
	if( IsPGPError( err ) )
		return err;
	pgpKeyDBChanged( keys, TRUE );
	(void)PGPCalculateTrust (keys->rootSet, NULL);
	return err;
}


/*
 *	Add a new User ID to a key.  User IDs cannot be added to other than the 
 *	user's own keys.  The new User ID is added to the end of the list.  To
 *	make it the primary User ID, call PGPSetPrimaryUserID() below.
 */

	PGPError
pgpAddUserID_internal (
	PGPKeyDBObjRef	key,
	PGPBoolean		isAttribute,
	PGPAttributeType attributeType,
	char const *	userIDData,
	PGPSize		   	userIDLength,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal )
{
	PGPKeyDB			*keys;
	PGPContextRef		 context;
	PGPSigSpec			*sigspec;
	PGPKeyDBObj			*userid;
	PGPByte const		*udata;
	PGPSize				 len;
	PGPUInt32			 otherattribute;
	PGPError			 error;

	error	= pgpKeyDeadCheck( key) ;
	if ( IsPGPError( error ) )
		return error;
	keys = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keys );

	/*  Can only add User ID to our own keys */
	if (!pgpKeyIsSec (key)) 
		return kPGPError_SecretKeyNotFound;

	/* See if we have a matching userid */
	for (userid = key->down; IsntNull( userid ); userid = userid->next)
	{
		if (!pgpKeyDBObjIsReal( userid) )
			continue;
		if( pgpObjectType( userid ) != RINGTYPE_USERID )
			continue;
		if( pgpUserIDIsAttribute( userid ) != isAttribute )
			continue;
		if( !isAttribute )
		{
			udata = (PGPByte const *)pgpUserIDName( userid, &len );
			if( len == userIDLength &&
							pgpMemoryEqual( udata, userIDData, len ) )
				return kPGPError_DuplicateUserID;
		} else {
			udata = pgpUserIDAttributeSubpacket( userid, 0,
												&otherattribute, &len, &error);
			if( IsntNull( udata ) && len == userIDLength &&
							otherattribute == (PGPUInt32)attributeType &&
							pgpMemoryEqual( udata, userIDData, len ) )
				return kPGPError_DuplicateUserID;
		}
	}

	if (isAttribute)
		userid = pgpCreateAttribute (keys, key, (PGPByte)attributeType,
									 (PGPByte *)userIDData, userIDLength);
	else
		userid = pgpCreateUserID (keys, key, (PGPByte *) userIDData,
								  userIDLength);

	if( IsNull( userid ) )
	{
		error = pgpKeyDBError( keys );
		return error;
	}

	/* Must self-certify here */
	error = sCreateSigSpec( context, key, PGP_SIGTYPE_KEY_GENERIC, passphrase,
							passphraseLength, hashedPhrase, cacheTimeOut,
							cacheGlobal, &sigspec );
	if( IsntPGPError( error ) )
		error = sSigSpecSetExportability( sigspec, SIG_EXPORTABLE, 0 );

	if( IsntPGPError( error ) )
		error = sCertifyObject( sigspec, userid );
	if( IsPGPError(error) ) {
		return error;
	}

	return error;
} 


static const PGPOptionType adduserOptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase
};

	PGPError
pgpAddUserIDInternal(
	PGPKeyDBObjRef		key,
	char const *		userID,
	PGPOptionListRef	optionList
	)
{
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPSize				userIDLength;
	PGPError			err = kPGPError_NoErr;

	pgpa(pgpaPGPKeyValid(key));
	PGPValidateKey( key );
	
	if( IsNull( userID ) )
		return kPGPError_BadParams;

	userIDLength = strlen(userID);

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						adduserOptionSet, elemsof( adduserOptionSet ) ) ) )
		return err;

	/* Pick up optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( pgpFrontEndKey( key ) )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( key );

		err = pgpAddUserID_back( PGPPeekKeyDBObjContext(key),
								 pgpKeyDBObjID( key ), FALSE,
								 (PGPAttributeType)0, userID, userIDLength,
								 passphrase, passphraseLength,
								 hashedPhrase, cacheTimeOut, cacheGlobal,
								 &newobjs, &newobjslen );
		if( IsPGPError( err ) )
			return err;
		err = pgpAddFromKeyArray( keydb, NULL, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
	} else {
		err = pgpAddUserID_internal( key, FALSE, (PGPAttributeType)0, userID,
								userIDLength, passphrase, passphraseLength,
								hashedPhrase, cacheTimeOut, cacheGlobal );
	}
error:
	return err;
}


static const PGPOptionType addattrOptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase
};

	PGPError
pgpAddAttributeInternal(
	PGPKeyDBObjRef		key,
	PGPAttributeType	attributeType,
	PGPByte	const	   *attributeData,
	PGPSize				attributeLength,
	PGPOptionListRef	optionList
	)
{
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;

	pgpa(pgpaPGPKeyValid(key));
	PGPValidateKey( key );
	
	if( IsNull( attributeData ) )
		return kPGPError_BadParams;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						adduserOptionSet, elemsof( adduserOptionSet ) ) ) )
		return err;

	/* Pick up optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( pgpFrontEndKey( key ) )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( key );

		err = pgpAddUserID_back( PGPPeekKeyDBContext(keydb),
								 pgpKeyDBObjID( key ), TRUE,
								 attributeType,
								 (const char *)attributeData, attributeLength,
								 passphrase, passphraseLength,
								 hashedPhrase, cacheTimeOut, cacheGlobal,
								 &newobjs, &newobjslen );
		if( IsPGPError( err ) )
			return err;
		err = pgpAddFromKeyArray( keydb, NULL, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
	} else {
		err = pgpAddUserID_internal( key, TRUE, attributeType,
									(const char *)attributeData,
									attributeLength, passphrase,
									passphraseLength, hashedPhrase,
									cacheTimeOut, cacheGlobal  );
	}
error:
	return err;
}

/* Raise specified userid object to be the first userid child of key */
static void
sRaiseName( PGPKeyDBObj *userid, PGPKeyDBObj *key )
{
	PGPKeyDBObj		  *obj;
	PGPKeyDBObj		  *predUserid = NULL; /* last child before userids */

	pgpAssert( userid->up == key );

	for( obj = key->down; IsntNull( obj ); obj = obj->next )
	{
		if( pgpObjectType( obj ) != RINGTYPE_USERID	)
		{
			predUserid = obj;
			continue;
		}

		/* Have a userid */
		if( IsNull(predUserid) || predUserid->next == obj )
		{
			/* First userid */
			if ( userid == obj )
				break;			/* Already first userid */
		}

		if( obj->next == userid )
		{
			obj->next = userid->next;
			if( IsNull( predUserid ) )
			{
				/* Userid becomes first child */
				userid->next = key->down;
				key->down = userid;
			} else {
				/* Userid goes after predUserid */
				userid->next = predUserid->next;
				predUserid->next = userid;
			}
			break;
		}
	}
}


/* This is used to just raise a userid */
	static PGPError
pgpSetPrimaryUserID_internal (PGPKeyDBObj *userid)
{
	PGPKeyDB		  *keys;
	PGPKeyDBObj		  *key;
	PGPUInt32		   attributeType = 0;
	PGPSize			   dummy;
	PGPError		   error;

	PGPValidateUserID( userid );
	CHECKREMOVED(userid);

	keys = PGPPeekKeyDBObjKeyDB( userid );
	key = userid->up;
	pgpAssert( pgpObjectType( key ) == RINGTYPE_KEY );
	error	= pgpKeyDeadCheck( key) ;
	if ( IsPGPError( error ) )
		return error;
	
	if( pgpUserIDIsAttribute( userid ) )
		(void)pgpUserIDAttributeSubpacket( userid, 0, &attributeType, &dummy,
										   &error );
	error = kPGPError_NoErr;

	/* Raise it to the top */
	sRaiseName( userid, key );

	/* Do nothing more if already the top */
	if( userid == pgpKeyPrimaryUserID( key, attributeType ) )
		return error;

	/* Else set the primary flag */
	pgpKeySetPrimaryUserID( userid );

	pgpKeyDBChanged(keys, TRUE);
	return kPGPError_NoErr;
}


/*	Make the given User ID the primary User ID of the key */
/*	This version uses OpenPGP Primary UserID signature subpackets */

	PGPError
pgpCertifyPrimaryUserID_internal (PGPKeyDBObj *userid,
	char *passphrase, PGPSize passphraseLength, PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal )
{
	PGPContextRef	   context;
	PGPKeyDB		  *keys;
	PGPKeyDBObj		  *key;
	PGPEnv			  *pgpEnv;
	PGPKeyDBObj		  *otherID;
	PGPKeyDBObj		  *bestsig;
	PGPSigSpec		  *sigspec;
	PGPBoolean		   wasprimary;
	PGPError		   error;
	int				   tzFix;

	/* If not our key, just raise it to the top */
	if( !pgpKeyIsSec (userid->up) )
		return pgpSetPrimaryUserID_internal( userid );

	keys = PGPPeekKeyDBObjKeyDB( userid );
	context = PGPPeekKeyDBContext( keys );
	pgpEnv = pgpContextGetEnvironment( context );
	tzFix  = pgpenvGetInt (pgpEnv, PGPENV_TZFIX, NULL, NULL);
	key    = userid->up;
	pgpAssert( pgpObjectType( key ) == RINGTYPE_KEY );

	/* Verify passphrase */
	if( !pgpSecPassphraseOK( key, (PGPByte *) passphrase, passphraseLength,
							 hashedPhrase, cacheTimeOut, cacheGlobal ) )
		return kPGPError_BadPassphrase;

	/*
	 * For each name, if other than the selected one, remove any primary
	 * userid subpackets.  For selected one, add one.
	 */
	for( otherID = key->down; IsntNull( otherID ) ; otherID = otherID->next )
	{
		if (!pgpKeyDBObjIsReal( otherID) )
			continue;
		if( pgpObjectType( otherID ) != RINGTYPE_USERID )
			continue;
		if( otherID == userid )
		{
			/* Name which is becoming primary */
			bestsig = pgpLatestSigByKey (otherID, key);
			if( IsntNull( bestsig ) )
			{
				sigspec = pgpSigSigSpec (bestsig, &error);
				if( IsNull( sigspec ) )
					continue;
				pgpSigSpecSetTimestamp( sigspec, pgpTimeStamp (tzFix) );
			}
			else
			{
				/* No previous sig, must create one */
				sigspec = pgpSigSpecCreate (pgpEnv, key,
											PGP_SIGTYPE_KEY_GENERIC);
				if( IsNull( sigspec ) )
					continue;
			}
			pgpSigSpecSetPrimaryUserID (sigspec, 0, TRUE);
			pgpSigSpecSetPassphrase( sigspec, (PGPByte *) passphrase,
							passphraseLength, hashedPhrase );
			error = pgpSignObject (otherID, sigspec);
			pgpSigSpecDestroy (sigspec);
			if( IsntPGPError( error ) && IsntNull( bestsig ) )
			{
				if ( IsPGPError( error = pgpMarkKeyDBObjectDeleted (bestsig)))
					return error;
			}
			error = kPGPError_NoErr;
		}
		else
		{
			/* Another name, must make sure it is no longer primary */
			bestsig = pgpLatestSigByKey (otherID, key);
			if( IsNull( bestsig ) )
				continue;
			sigspec = pgpSigSigSpec (bestsig, &error);
			if( IsNull( sigspec ) )
				continue;
			pgpSigSpecPrimaryUserID (sigspec, &wasprimary);
			if( !wasprimary )
			{
				pgpSigSpecDestroy (sigspec);
				continue;
			}
			pgpSigSpecSetTimestamp( sigspec, pgpTimeStamp (tzFix) );
			pgpSigSpecSetPrimaryUserID (sigspec, 0, FALSE);
			pgpSigSpecSetPassphrase( sigspec, (PGPByte *) passphrase,
							passphraseLength, hashedPhrase );
			error = pgpSignObject (otherID, sigspec);
			pgpSigSpecDestroy (sigspec);
			if( IsntPGPError( error ) )
			{
				if( IsPGPError( error = pgpMarkKeyDBObjectDeleted (bestsig) ) )
					return error;
			}
			error = kPGPError_NoErr;
		}
	}

	/* Raise the name to the top */
	sRaiseName( userid, key );
	pgpKeyDBChanged(keys, TRUE);
	
	return error;

}


static const PGPOptionType setprimaryOptionSet[] = {
	 kPGPOptionType_Passphrase,
	 kPGPOptionType_Passkey,
	 kPGPOptionType_CachePassphrase
};

	PGPError
pgpCertifyPrimaryUserIDInternal (
	PGPKeyDBObj *		userid,
	PGPOptionListRef	optionList
	)
{
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;

	PGPValidateUserID( userid );
	
	CHECKREMOVED(userid);
	pgpAssert( pgpObjectType( userid->up ) == RINGTYPE_KEY );

	err = pgpKeyDeadCheck( userid->up );
	if ( IsPGPError( err ) )
		goto error;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
					setprimaryOptionSet, elemsof( setprimaryOptionSet ) ) ) )
		return err;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( pgpFrontEndKey( userid ) )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( userid );

		err = pgpCertifyPrimaryUserID_back( PGPPeekKeyDBContext(keydb),
										pgpKeyDBObjID(userid), passphrase,
										passphraseLength, hashedPhrase,
										cacheTimeOut, cacheGlobal,
										&newobjs, &newobjslen );
		if( IsPGPError( err ) )
			return err;
		pgpKeyDBObjRefresh( userid->up, TRUE );
		err = pgpAddFromKeyArray( keydb, NULL, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
	} else {
		err = pgpCertifyPrimaryUserID_internal( userid, passphrase,
											passphraseLength, hashedPhrase,
											cacheTimeOut, cacheGlobal );
	}

error:
	return err;
}


/*  Certify a User ID.  Do not allow duplicate certification. If UserID
    is already certified, but revoked, the old cert can
	be removed and the UserID then recertified. */

	PGPError
pgpCertifyUserID_internal(
	PGPKeyDBObj *	userid,
	PGPKeyDBObjRef	certifying_key,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPBoolean		exportable,
	PGPTime			creationDate,
	PGPUInt32		expiration,
	PGPByte			trustDepth,
	PGPByte			trustValue,
	char const *	sRegExp
	)
{
	PGPKeyDB		    *keys;
	PGPContextRef		 context;
	PGPSigSpec			*sigspec;
	PGPKeyDBObj		    *sig;
	PGPKeyDBObj			*parent_key;
	PGPError			 error = kPGPError_NoErr;

	parent_key = userid->up;
	error	= pgpKeyDeadCheck( parent_key ) ;
	if ( IsPGPError( error ) )
		return error;

	pgpAssert( pgpKeyIsValid(certifying_key) );

	keys = PGPPeekKeyDBObjKeyDB( userid );
	context = PGPPeekKeyDBContext( keys );

	error	= pgpKeyDeadCheck( certifying_key ) ;
	if ( IsPGPError( error ) )
		return error;

	/*  Check for duplicate certificate.  There may be some
		old revocation certs still laying around, which we
		should ignore.  */

	for( sig = userid->down; IsntNull( sig ); sig = sig->next )
	{
		if (!pgpKeyDBObjIsReal( sig) )
			continue;
		if( pgpObjectType( sig ) != RINGTYPE_SIG )
			continue;
		if (pgpSigMaker( sig ) == certifying_key &&
			pgpSigType( sig ) != PGP_SIGTYPE_KEY_UID_REVOKE)
		{
		    error = kPGPError_DuplicateCert;
			break;
		}
	}
	if (error)
	    return error;

	error = sCreateSigSpec( context, certifying_key, PGP_SIGTYPE_KEY_GENERIC,
							passphrase, passphraseLength, hashedPhrase,
							cacheTimeOut, cacheGlobal, &sigspec );
	if( IsntPGPError( error ) )
		error = sSigSpecSetExportability( sigspec, exportable,
										  SIG_EXPORTABLEHASHED );
	if( IsntPGPError( error ) )
		error = sSigSpecSetTimes( sigspec, creationDate, expiration );
	if( IsntPGPError( error ) )
		error = sSigSpecSetTrustParams( sigspec, keys, trustDepth, trustValue,
										sRegExp );

	if( IsntPGPError( error ) )
		error = sCertifyObject( sigspec, userid );

	return error;
}


static const PGPOptionType signuserOptionSet[] = {
	 kPGPOptionType_Passphrase,
	 kPGPOptionType_Passkey,
	 kPGPOptionType_CachePassphrase,
	 kPGPOptionType_Expiration,
	 kPGPOptionType_CreationDate,
	 kPGPOptionType_Exportable,
	 kPGPOptionType_CertificateTrust,
	 kPGPOptionType_CertificateRegularExpression
};

	PGPError
pgpCertifyUserIDInternal (
	PGPKeyDBObj *		userid,
	PGPKeyDBObj *		certifying_key,
	PGPOptionListRef	optionList
	)
{
	PGPContextRef		context;
	PGPKeyDB *			keys;
	PGPTime				creationDate;
	PGPUInt32			expiration;
	PGPUInt32			exportable;
	PGPUInt32			trustDepth;
	PGPUInt32			trustLevel;
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	char const *		sRegExp;
	PGPSize				sRegExpLength = 0;
	PGPError			err = kPGPError_NoErr;

	keys = PGPPeekKeyDBObjKeyDB( userid );
	context = PGPPeekKeyDBContext( keys );

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						signuserOptionSet, elemsof( signuserOptionSet ) ) ) )
		return err;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;
		
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_CreationDate, FALSE,
						 "%T", &creationDate ) ) )
		goto error;
		
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Expiration, FALSE,
						 "%d", &expiration ) ) )
		goto error;
		
	/* Defaults exportable to false */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Exportable, FALSE,
						 "%d", &exportable ) ) )
		goto error;
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_CertificateTrust, FALSE,
						 "%d%d", &trustDepth, &trustLevel ) ) )
		goto error;
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_CertificateRegularExpression, FALSE,
						 "%p", &sRegExp ) ) )
		goto error;

	/* Check regexp for validity */
	if( IsntNull( sRegExp ) ) {
		regexp *rexp;
		sRegExpLength = strlen(sRegExp) + 1;
		if (IsPGPError( pgpRegComp( context, sRegExp, &rexp ) ) ) {
			pgpDebugMsg( "Invalid regular expression" );
			err = kPGPError_BadParams;
			goto error;
		}
		pgpContextMemFree( context, rexp );
	}

	/* Expiration is given as days from today, we will convert to seconds */
	if( expiration != 0 )
		expiration *= (24*60*60);

	if( pgpFrontEndKey( userid ) )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( userid );

		err = pgpCertifyUserID_back( PGPPeekKeyDBContext(keydb),
							pgpKeyDBObjID( userid ),
							pgpKeyDBObjID(certifying_key), passphrase,
							passphraseLength, hashedPhrase, cacheTimeOut,
							cacheGlobal, (PGPBoolean)exportable,
							creationDate, expiration,
							(PGPByte) trustDepth, (PGPByte) trustLevel,
							sRegExp, sRegExpLength, &newobjs, &newobjslen );
		if( IsPGPError( err ) )
			return err;
		pgpKeyDBObjRefresh( userid->up, TRUE );
		err = pgpAddFromKeyArray( keydb, userid->up, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
		if( PGPPeekKeyDBObjKeyDB( certifying_key ) != keydb )
		{
			/* On a cross keydb certification we must do a refresh in case
			 * we created some dummy keys
			 */
			PGPUInt32		numKeys;
			PGPUInt32 *		keyArray;
			PGPSize			keyArraySize;
			err = pgpKeyDBArray_back( PGPPeekKeyDBContext(keydb),
									  keydb->id,
									  &numKeys, &keyArray, &keyArraySize );
			if( IsPGPError( err ) )
				return err;
			err = pgpAddFromKeyArray( keydb, NULL, keyArray, numKeys, FALSE );
			if( IsntNull( keyArray ) )
				PGPFreeData( keyArray );
			if( IsPGPError( err ) )
				return err;
		}
	} else {
		err = pgpCertifyUserID_internal( userid, certifying_key,
							  passphrase, passphraseLength, hashedPhrase,
							  cacheTimeOut, cacheGlobal,
							  (PGPBoolean)exportable, creationDate, expiration,
							  (PGPByte) trustDepth, (PGPByte) trustLevel,
							  sRegExp );
	}

	/* Calculate trust changes as a result */
	if( err == kPGPError_NoErr )
		(void)PGPCalculateTrust (PGPPeekKeyDBObjKeyDB(userid)->rootSet, NULL);

error:
	return err;
}


/*  Given a cert, return the certifying key object.  The signing key does not 
	have to be in the same keydb as <sig>, and may be in the <other> keydb. */

	PGPError
pgpGetSigCertifierKey(
	PGPKeyDBObjRef		sig,
	PGPKeyDBRef			otherdb,
	PGPKeyDBObj **		certkey)
{
	PGPKeyDBObj *			key;
	PGPKeyID				keyID;
	PGPError				err	= kPGPError_NoErr;

	/* See if we have a copy of the signer in the same db */
	key = pgpSigMaker( sig );
	if( IsntNull( key ) )
	{
		*certkey = key;
		return kPGPError_NoErr;
	}
	
	/* Must check other keydb using keyid */
	if( IsNull( otherdb ) )
		return kPGPError_ItemNotFound;

	err	= pgpGetKeyIDOfCertifier( sig, &keyID);
	if( IsPGPError( err ) )
		return err;

	err	= PGPFindKeyByKeyID(otherdb, &keyID, certkey );
	
	pgpAssertErrWithPtr( err, *certkey );
	return err;
}


	PGPError
PGPGetSigCertifierKey (
	PGPKeyDBObjRef		sig,
	PGPKeyDBRef			otherdb,
	PGPKeyDBObj **		certkey)
{

	PGPValidatePtr( certkey );
	*certkey	= NULL;
	PGPValidateSig( sig );
	if( IsntNull( otherdb ) )
		PGPValidateKeyDB( otherdb );

	pgpEnterPGPErrorFunction();
	
	CHECKREMOVED(sig);
	
	return pgpGetSigCertifierKey( sig, otherdb, certkey );
}



/*  Given an X.509 sig, return the certifying sig object.
 *  The signing sig does not 
 *	have to be in the same keydb as <sig>, and may be in the <other> keydb.
 */
	PGPError
PGPGetSigX509CertifierSig (
	PGPKeyDBObjRef		sig,
	PGPKeyDBRef			otherdb,
	PGPKeyDBObj **		certsig)
{
	PGPKeyDBRef				keydb;
	PGPContextRef			context;
	PGPBoolean				isX509;
	PGPKeyDBObjRef			certkey;
	PGPKeyDBObjRef			userid;
	PGPKeyDBObjRef			usig;
	PGPByte				   *issuername = NULL;
	PGPSize					issuernamelen;
	PGPByte				   *signame = NULL;
	PGPSize					signamelen;
	PGPBoolean				match = FALSE;
	PGPError				err	= kPGPError_NoErr;

	PGPValidatePtr( certsig );
	*certsig	= NULL;

	PGPValidateSig( sig );
	if( IsntNull( otherdb ) )
		PGPValidateKeyDB( otherdb );
	
	pgpEnterPGPErrorFunction();

	CHECKREMOVED(sig);
	
	keydb = PGPPeekKeyDBObjKeyDB( sig );
	context = PGPPeekKeyDBContext( keydb );
	
	err = pgpGetSigBoolean (sig, kPGPSigProperty_IsX509, &isX509);
	if( IsPGPError( err ) )
		goto error;
	if( !isX509 )
	{
		err = kPGPError_BadParams;
		goto error;
	}

	err = pgpGetSigCertifierKey( sig, otherdb, &certkey );
	if( IsPGPError( err ) )
		goto error;

	err = pgpGetSigPropertyBuffer( sig, kPGPSigProperty_X509IssuerLongName,
								   0, NULL, &issuernamelen );
	if( IsPGPError( err ) )
		goto error;

	issuername = pgpContextMemAlloc( context, issuernamelen, 0 );
	if( IsNull( issuername ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}

	err = pgpGetSigPropertyBuffer( sig, kPGPSigProperty_X509IssuerLongName,
								   issuernamelen, issuername, &issuernamelen );
	if( IsPGPError( err ) )
		goto error;

	for( userid = certkey->down; IsntNull(userid) && !match;
		 									userid = userid->next )
	{
		if (!pgpKeyDBObjIsReal( userid ) )
			continue;
		if( pgpObjectType( userid ) != RINGTYPE_USERID )
			continue;
		for( usig = userid->down; IsntNull(usig) && !match;
			 									usig = usig->next )
		{
			/* Find a non-removed, X509 sig which matches issuername */
			if (!pgpKeyDBObjIsReal( usig ) )
				continue;
			if( pgpObjectType( usig ) != RINGTYPE_SIG )
				continue;
			err = pgpGetSigBoolean (usig, kPGPSigProperty_IsX509, &isX509);
			if( IsPGPError( err ) )
				goto error;
			if( !isX509 )
				continue;
			err = pgpGetSigPropertyBuffer( usig, kPGPSigProperty_X509LongName,
										   0, NULL, &signamelen );
			if( IsPGPError( err ) )
				goto error;
			if( signamelen != issuernamelen )
				continue;
			signame = pgpContextMemAlloc( context, signamelen, 0 );
			if( IsNull( signame ) )
				{
					err = kPGPError_OutOfMemory;
					goto error;
				}
			err = pgpGetSigPropertyBuffer( usig, kPGPSigProperty_X509LongName,
										   signamelen, signame,
										   &signamelen );
			if( IsPGPError( err ) )
				goto error;

			match = pgpMemoryEqual( issuername, signame, signamelen );
			pgpContextMemFree( context, signame );
			signame = NULL;
			if( match )
				*certsig = usig;
		}
	 }


error:

	if( IsntNull( issuername ) )
		pgpContextMemFree( context, issuername );
	if( IsntNull( signame ) )
		pgpContextMemFree( context, signame );

	return err;
}


/*  Revoke a signature.  */

	PGPError
pgpRevokeSig_internal (
	PGPContextRef	context,
	PGPKeyDBObjRef	sig,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal
	)
{
	PGPKeyDB			*keys;
	PGPSigSpec			*sigspec;
	PGPKeyDBObj			*signkey;
	PGPKeyDBObj			*userid;
	PGPBoolean			 revoked;
	PGPBoolean			 exportable;
	PGPError			 error = kPGPError_NoErr;

	CHECKREMOVED(sig);

	keys = PGPPeekKeyDBObjKeyDB( sig );

	error = pgpGetSigBoolean (sig, kPGPSigProperty_IsRevoked, &revoked);
	if (error)
		return error;
	if (revoked)
		return kPGPError_NoErr;   /* already revoked */

	/*  Get certifying key */
	error = pgpGetSigCertifierKey (sig, NULL, &signkey);
	if (error)
		return error;
	if( IsNull( signkey ) )
		return kPGPError_SecretKeyNotFound;

	userid = sig->up;
	pgpAssert( pgpObjectType( userid ) == RINGTYPE_USERID );

	/* Copy exportability attribute from cert we are revoking */
	error = pgpGetSigBoolean (sig, kPGPSigProperty_IsExportable, &exportable);
	if (error)
		return error;
	error = sCreateSigSpec( context, signkey, PGP_SIGTYPE_KEY_UID_REVOKE,
							passphrase, passphraseLength, hashedPhrase,
							cacheTimeOut, cacheGlobal, &sigspec );
	if( IsntPGPError( error ) )
		error = sSigSpecSetExportability( sigspec, exportable,
										  SIG_EXPORTABLEHASHED );

	if( IsntPGPError( error ) )
		error = sCertifyObject( sigspec, userid );

	return error;
}


static const PGPOptionType revsigOptionSet[] = {
	 kPGPOptionType_Passphrase,
	 kPGPOptionType_Passkey,
	 kPGPOptionType_CachePassphrase
};

	PGPError
pgpRevokeSigInternal(
	PGPKeyDBObjRef		sig,
	PGPOptionListRef	optionList
	)
{
	PGPContextRef		context;
	PGPKeyDBRef			keys;
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;

	pgpa(pgpaPGPCertValid(sig));
	PGPValidateSig( sig );

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						revsigOptionSet, elemsof( revsigOptionSet ) ) ) )
		return err;

	/* Pick up optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	keys = PGPPeekKeyDBObjKeyDB( sig );
	context = PGPPeekKeyDBContext( keys );

	if( pgpFrontEndKey( sig ) )
	{
		PGPUInt32 *newobjs;
		PGPKeyDBObj *gparent;
		PGPSize newobjslen;
		PGPKeyDB *keydb = PGPPeekKeyDBObjKeyDB( sig );

		err = pgpRevokeSig_back(context, pgpKeyDBObjID(sig), passphrase,
								passphraseLength, hashedPhrase, cacheTimeOut,
								cacheGlobal, &newobjs, &newobjslen);
		if( IsPGPError( err ) )
			return err;
		gparent = OBJISTOPKEY(sig->up) ? NULL : sig->up->up;
		pgpKeyDBObjRefresh( PGPPeekKeyDBObjKey(sig), TRUE );
		err = pgpAddFromKeyArray( keydb, gparent, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
	} else {
		err = pgpRevokeSig_internal(context, sig, passphrase,
									passphraseLength, hashedPhrase,
									cacheTimeOut, cacheGlobal );
	}

	/* Calculate trust changes as a result */
	if( err == kPGPError_NoErr )
		(void)PGPCalculateTrust (PGPPeekKeyDBObjKeyDB(sig)->rootSet, NULL);

error:
	return err;
}



	PGPError
PGPCountAdditionalRecipientRequests(
	PGPKeyDBObjRef		basekey,
    PGPUInt32 *			numARKeys)
{
	PGPUInt32			 nadks;			/* Number ADK's available */
	PGPByte	 			tclass;			/* Class code from ADK */
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidatePtr( numARKeys );
	*numARKeys	= 0;
	PGPValidateKey( basekey );
	
	pgpEnterPGPErrorFunction();

	(void)pgpKeyAdditionalRecipientRequestKey (basekey, 0,
										NULL, NULL, &tclass, &nadks, &err);
	if ( err == kPGPError_ItemNotFound )
	{
		nadks	= 0;
		err		= kPGPError_NoErr;
	}
	
	*numARKeys = nadks;
	return err;
}


/*  Return the nth (0 based) additional decryption key and keyid,
	if one exists.
	It is an error to use an index >= K, where K is the number of ARR key ids.
	
 	Also return the class of the ADK.  The class is currently reserved
 	for use by PGP.
 	Any of the return pointers may be NULL.
 	
	Note that it is *not* safe to use the keyID returned from this function
	to get the ADK to use because KeyIDs are not unique.
	Instead, the keyID can be used to locate the actual key(s) with that
	key id.
	Then call this function again to get the ADK;
	it will check the key fingerprint, which is unique.

*/
	static PGPError
pgpGetIndexedAdditionalRecipientRequestKey(
	PGPKeyDBObjRef		basekey,
	PGPUInt32			nth,
    PGPKeyDBObjRef*		adkey,
	PGPKeyID *			adkeyid,
    PGPByte *			adclass)
{
	PGPKeyDB *			keys;			/* Key DB */
	PGPKeyDBObj *		rkey;			/* Aurora additional decryption key */
	unsigned			nadks;			/* Number ADK's available */
	PGPByte				tclass;			/* Class code from ADK */
	PGPError			error;			/* Error return from Aurora */
	PGPByte				pkalg;			/* pkalg of ADK */
	PGPKeyID			keyid;			/* keyid of ADK */
	PGPError			 err	= kPGPError_NoErr;

	if( IsntNull( adkeyid ) )
		pgpClearMemory( adkeyid, sizeof( *adkeyid ) );
	if ( IsntNull( adclass ) )
		*adclass	= 0;
	if ( IsntNull( adkey ) )
		*adkey	= NULL;
		
	PGPValidateKey( basekey );
	
	keys = PGPPeekKeyDBObjKeyDB( basekey );

	rkey = pgpKeyAdditionalRecipientRequestKey (basekey, nth,
								 &pkalg, &keyid, &tclass, &nadks, &error);

	if( IsPGPError( error ) )
	{
		return error;
	}
	
	/* Success */
	if ( IsntNull( adkey ) )
	{
		*adkey = rkey;
	}

	if ( IsntNull( adkeyid ) )
	{
		*adkeyid	= keyid;
	}
	
	if ( IsntNull( adclass ) )
		*adclass = tclass;

	return err;
}



/* Given a key, return the nth (0 based) additional decryption key, if
 	one exists.  Also return the keyid, the class of the ADK, and the
 	number of ADK's for the base key.  Any of the return pointers may
 	be NULL. */

	PGPError
PGPGetIndexedAdditionalRecipientRequestKey(
	PGPKeyDBObjRef		basekey,
	PGPUInt32			nth,
    PGPKeyDBObjRef *	adkey,
	PGPKeyID *			adkeyid,
    PGPByte *			adclass)
{
	PGPError	err	= kPGPError_NoErr;
	PGPKeyID	tempKeyID;
	
	if ( IsntNull( adkey ) )
		*adkey	= NULL;
	if ( IsntNull( adkeyid ) )
		pgpClearMemory( adkeyid, sizeof( *adkeyid) );
	if ( IsntNull( adclass ) )
		*adclass	= 0;

	PGPValidateKey( basekey );
	
	pgpEnterPGPErrorFunction();

	err	= pgpGetIndexedAdditionalRecipientRequestKey( basekey,
			nth, adkey, &tempKeyID, adclass );
	if ( IsntPGPError( err ) )
	{
		pgpAssert( pgpKeyIDIsValid( &tempKeyID ) );
		if( IsntNull( adkeyid ) )
		{
			*adkeyid	= tempKeyID;
		}
	}
	else
	{
		pgpClearMemory( adkeyid, sizeof( *adkeyid) );
	}
	
	return( err );
}


	PGPError
PGPCountRevocationKeys(
	PGPKeyDBObjRef		basekey,
    PGPUInt32 *			numRevKeys)
{
	PGPUInt32			nrevs;			/* Number rev keys available */
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidatePtr( numRevKeys );
	*numRevKeys	= 0;
	PGPValidateKey( basekey );
	
	pgpEnterPGPErrorFunction();

	(void)pgpKeyRevocationKey (basekey, 0, NULL, NULL, NULL, &nrevs, &err);
	if ( err == kPGPError_ItemNotFound )
	{
		nrevs	= 0;
		err		= kPGPError_NoErr;
	}
	
	*numRevKeys = nrevs;
	return err;
}


/*  Return the nth (0 based) revocation key and keyid,
	if one exists.
	It is an error to use an index >= K, where K is the number of ARR key ids.
	
 	Also return the class of the revkey.  The high bit is set for a
	reocation key.
 	Any of the return pointers may be NULL.
 	
	Note that it is *not* safe to use the keyID returned from this function
	to get the revkey to use because KeyIDs are not unique.
	Instead, the keyID can be used to locate the actual key(s) with that
	key id.
	Then call this function again to get the revkey;
	it will check the key fingerprint, which is unique.

*/
	static PGPError
pgpGetIndexedRevocationKey(
	PGPKeyDBObjRef		basekey,
	PGPUInt32			nth,
    PGPKeyDBObjRef *	revkey,
	PGPKeyID *			revkeyid,
    PGPByte *			revclass)
{
	PGPKeyDB *			keys;			/* Key DB */
	PGPKeyDBObj *		rkey;			/* Aurora revocation key */
	unsigned			nrevks;			/* Number revkey's available */
	PGPByte				tclass;			/* Class code from revkey */
	PGPError			error;			/* Error return from Aurora */
	PGPByte				pkalg;			/* pkalg of revkey */
	PGPKeyID			keyid;			/* keyid of revkey */
	PGPError			err	= kPGPError_NoErr;

	if( IsntNull( revkeyid ) )
		pgpClearMemory( revkeyid, sizeof( *revkeyid ) );
	if ( IsntNull( revclass ) )
		*revclass	= 0;
	if ( IsntNull( revkey ) )
		*revkey	= NULL;
		
	PGPValidateKey( basekey );
	
	keys = PGPPeekKeyDBObjKeyDB( basekey );

	rkey = pgpKeyRevocationKey (basekey, nth,
								&pkalg, &keyid, &tclass, &nrevks, &error);

	if( IsPGPError( error ) )
	{
		return error;
	}
	
	/* Success */
	if ( IsntNull( revkey ) )
	{
		*revkey = rkey;
	}

	if ( IsntNull( revkeyid ) )
	{
		*revkeyid	= keyid;
	}
	
	if ( IsntNull( revclass ) )
		*revclass = tclass;

	return err;
}



/* Given a key, return the nth (0 based) revocation key, if
 	one exists.  Also return the keyid, the class of the revkey, and the
 	number of revkey's for the base key.  Any of the return pointers may
 	be NULL. */

	PGPError
PGPGetIndexedRevocationKey(
	PGPKeyDBObjRef		basekey,
	PGPUInt32			nth,
    PGPKeyDBObjRef *	revkey,
	PGPKeyID *			revkeyid)
{
	PGPError	err	= kPGPError_NoErr;
	PGPKeyID	tempKeyID;
	
	if ( IsntNull( revkey ) )
		*revkey	= NULL;
	if ( IsntNull( revkeyid ) )
		pgpClearMemory( revkeyid, sizeof( *revkeyid) );

	PGPValidateKey( basekey );
	
	pgpEnterPGPErrorFunction();

	err	= pgpGetIndexedRevocationKey( basekey, nth, revkey, &tempKeyID, NULL );
	if ( IsntPGPError( err ) )
	{
		pgpAssert( pgpKeyIDIsValid( &tempKeyID ) );
		if( IsntNull( revkeyid ) )
		{
			*revkeyid	= tempKeyID;
		}
	}
	else
	{
		pgpClearMemory( revkeyid, sizeof( *revkeyid) );
	}
	
	return( err );
}


/*
 * Return a buffer with CRL distribution points in it.  *pnDistPoints
 * tells how many distribution points there are; *pdpointLengths holds
 * the size of each distribution point; *pDpoints holds the actual
 * distribution point pointer.  The latter two values are dynamically
 * allocated and should be freed by the caller.
 */
	PGPError
PGPGetCRLDistributionPoints(
	PGPKeyDBObjRef cakey,
	PGPUInt32 *pnDistPoints,			/* Output parameters */
	PGPByte **pDpoints,
	PGPSize **pdpointLengths
	)
{
	PGPKeyDBRef			keys;
	PGPContextRef		context;
	PGPMemoryMgrRef		mgr;
	PGPUInt32			nDistPoints;
	PGPByte				*dpoints;
	PGPSize				*dpointlens;
	PGPError			error = kPGPError_NoErr;

	if ( IsntNull( pnDistPoints ) )
		*pnDistPoints = 0;
	if ( IsntNull( pDpoints ) )
		*pDpoints = NULL;
	if ( IsntNull( pdpointLengths ) )
		*pdpointLengths = NULL;

	PGPValidateKey( cakey );
	
	pgpEnterPGPErrorFunction();

	keys = PGPPeekKeyDBObjKeyDB( cakey );
	context = PGPPeekKeyDBContext( keys );
	mgr = PGPPeekContextMemoryMgr( context );
	
	error = pgpListCRLDistributionPoints( mgr, cakey,
									&nDistPoints, &dpoints, &dpointlens );
	if( IsPGPError( error ) )
	{
		return error;
	}

	if ( IsntNull( pnDistPoints ) )
		*pnDistPoints = nDistPoints;
	if ( IsntNull( pDpoints ) )
		*pDpoints = dpoints;
	if ( IsntNull( pdpointLengths ) )
		*pdpointLengths = dpointlens;

	return kPGPError_NoErr;
}


/*  Trust-related functions */

#if 0	/* KEEP [ */
	PGPError
PGPSetUserIDConfidence(PGPKeyDBObj *userid, PGPUInt32 confidence)
{
	PGPKeyDB			*keys;
	PGPKeyDBObj			*key;
	PGPKeyDBObj			*userid;
	PGPError			 error = kPGPError_NoErr;

	pgpa(pgpaPGPUserIDValid(userid));
	PGPValidateUserID( userid );
	
	pgpEnterPGPErrorFunction();

	keys = PGPPeekKeyDBObjKeyDB( userid );

	CHECKREMOVED(userid);
	key = userid->up;
	pgpAssert( pgpObjectType( key ) == RINGTYPE_KEY );

	err	= pgpKeyDeadCheck( key );
	if ( IsPGPError( err ) )
		return err;

	if (pgpKeyIsSec ( key ) )
		return kPGPError_BadParams;

	pgpUserIDSetConfidence ( userid, (unsigned short) confidence);
	pgpKeyDBChanged (keys, TRUE);

cleanup:
	return error;
}

#endif	/* ] KEEP */

/*  Set the trust on a key.  Cannot be used to set undefined or 
	axiomatic trust.   The key must be valid to assign trust. */

	PGPError
pgpSetKeyTrust_internal (PGPKeyDBObj *key, PGPUInt32 trust)
{
	PGPEnv				*pgpEnv;
	PGPContextRef		context;
	PGPKeyDB			*keys;
	PGPKeyDBObj			*userid;
	PGPError			 error = kPGPError_NoErr;
	PGPUInt16			 confidence;
#if ONLY_TRUST_VALID_KEYS
	long                 validity;
#endif

	keys = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keys );
	pgpEnv = pgpContextGetEnvironment( context );

#if ONLY_TRUST_VALID_KEYS
	/*  Should not set trust on key that is not completely valid 
		(who is it we are trusting?) */
	PGPGetKeyNumber (key, kPGPKeyPropValidity, &validity);
	if (validity != kPGPValidity_Complete) 
	    return kPGPError_BadParams;
#endif

	confidence = pgpTrustToIntern(
		(PGPByte)pgpTrustOldToExtern( pgpEnv, (PGPByte)trust ) );

	for( userid = key->down; IsntNull( userid ); userid = userid->next )
	{
		if (!pgpKeyDBObjIsReal( userid ) )
			continue;
		if( pgpObjectType( userid ) != RINGTYPE_USERID )
			continue;
		pgpUserIDSetConfidence( userid, (unsigned short) confidence );
	}

	/* Also set key trust for consistency */
	pgpKeySetTrust (key, (PGPByte)trust);

	pgpKeyDBChanged (keys, TRUE);

	return error;
}


	PGPError
PGPSetKeyTrust (PGPKeyDBObj *key, PGPUInt32 trust)
{
	PGPError			err;

	PGPValidateKey( key );
	
	pgpEnterPGPErrorFunction();

	if (trust <= kPGPKeyTrust_Undefined || trust > kPGPKeyTrust_Complete ||
		 pgpKeyAxiomatic ( key ))
	{
		return kPGPError_BadParams;
	}

	err	= pgpKeyDeadCheck( key);
	if ( IsPGPError( err ) )
		return err;

	if( pgpFrontEndKey( key ) )
	{
		err = pgpSetKeyTrust_back( PGPPeekKeyDBObjContext(key),
									pgpKeyDBObjID(key), trust );
		if( IsPGPError( err ) )
			return err;
		pgpKeyDBObjRefresh( key, FALSE );
		pgpKeyDBChanged (PGPPeekKeyDBObjKeyDB(key), TRUE);
	} else {
		err = pgpSetKeyTrust_internal( key, trust );
	}

	/* Calculate trust changes as a result */
	if( IsntPGPError( err ) )
		err = PGPCalculateTrust( PGPPeekKeyDBObjKeyDB(key)->rootSet, NULL );

	return err;
}


/*  Set a secret key as the axiomatic key.  If checkPassphrase == TRUE,
	the user must prove knowledge of the passphrase in order to do 
	this. */

	PGPError
pgpSetKeyAxiomatic_internal (
	PGPKeyDBObj *	key,
	PGPBoolean		setAxiomatic,
	PGPBoolean		checkPassphrase,
	char const *	passphrase,
	PGPSize		passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal
	)
{
	PGPBoolean               secret, axiomatic;
	PGPSecKey         		*seckey;
	PGPKeyDB                *keys;
	PGPContextRef			 context;
	PGPError                 error = kPGPError_NoErr;

	if( !setAxiomatic )
		return pgpUnsetKeyAxiomatic( key );

	pgpGetKeyBoolean (key, kPGPKeyProperty_IsSecret, &secret);
	if (!secret)
		return kPGPError_BadParams;
	pgpGetKeyBoolean (key, kPGPKeyProperty_IsAxiomatic, &axiomatic);
	if (axiomatic)
		return kPGPError_NoErr;

	keys = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keys );

	if (checkPassphrase) {
		/* Get the secret key and attempt to unlock it */
		seckey = pgpSecSecKey (key, PGP_PKUSE_SIGN);
		if (!seckey)
			return pgpKeyDBError(keys);
#if 1
		error = pgpSecKeyUnlockWithCache( seckey, (PGPByte const *)passphrase,
										passphraseLength, hashedPhrase,
										cacheTimeOut, cacheGlobal );
		pgpSecKeyDestroy( seckey );
		if( IsPGPError( error ) )
			return error;
#else
		if (pgpSecKeyIslocked (seckey)) {
			if (IsNull( passphrase )) {
				pgpSecKeyDestroy (seckey);
				return kPGPError_BadPassphrase;
			}
			error = (PGPError)pgpSecKeyUnlock (seckey, passphrase, 
									 passphraseLength, hashedPhrase);
			pgpSecKeyDestroy (seckey);
			if (error != 1) {
				if (error == 0)
					error = kPGPError_BadPassphrase;
				return error;
			}
		}
#endif
	}

	/*  Make sure it's enabled first before setting axiomatic */
	if ((error = pgpEnableKey (key)) != kPGPError_NoErr)
		return error;

	pgpKeySetAxiomatic( key );
	pgpKeyDBChanged (keys, TRUE);

	return error;
}


static const PGPOptionType setkeyaxiomaticOptionSet[] = {
	 kPGPOptionType_Passphrase,
	 kPGPOptionType_Passkey,
	 kPGPOptionType_CachePassphrase
};

	PGPError
pgpSetKeyAxiomaticInternal(
	PGPKeyDBObjRef		key,
	PGPBoolean			setAxiomatic,
	PGPOptionListRef	optionList
	)
{
	PGPKeyDB *			keys;
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;

	pgpa(pgpaPGPKeyValid(key));
	PGPValidateKey( key );
	keys = PGPPeekKeyDBObjKeyDB( key );

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
								setkeyaxiomaticOptionSet,
								elemsof( setkeyaxiomaticOptionSet ) ) ) )
		goto error;

	/* Pick up optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Passphrase, FALSE,
						 "%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( pgpFrontEndKey( key ) )
	{
		err = pgpSetKeyAxiomatic_back( PGPPeekKeyDBObjContext(key),
							pgpKeyDBObjID( key ), setAxiomatic,
							(PGPBoolean)(passphrase!=NULL),
							passphrase, passphraseLength, hashedPhrase,
							cacheTimeOut, cacheGlobal );

		if( IsntPGPError( err ) )
		{
			pgpKeyDBObjRefresh( key, FALSE );
			pgpKeyDBChanged (keys, TRUE);
		}
	} else {
		err = pgpSetKeyAxiomatic_internal( key, setAxiomatic,
							(PGPBoolean)(passphrase!=NULL),
							passphrase, passphraseLength, hashedPhrase,
							cacheTimeOut, cacheGlobal );

	}
	/* Calculate trust changes as a result */
	if( IsntPGPError( err ) )
		err = PGPCalculateTrust( keys->rootSet, NULL );

error:

	return err;
}



	PGPError
pgpUnsetKeyAxiomatic (PGPKeyDBObj *key)
{
	PGPBoolean				  axiomatic;
	PGPKeyDB                 *keys;
	PGPError                  error = kPGPError_NoErr;

	PGPValidateKey( key );

	keys = PGPPeekKeyDBObjKeyDB( key );

	pgpGetKeyBoolean (key, kPGPKeyProperty_IsAxiomatic, &axiomatic);
	if (!axiomatic)
		return kPGPError_BadParams;

	pgpKeyResetAxiomatic (key);
	pgpKeyDBChanged (keys, TRUE);

	return error;
}



/*  Get property functions.  Internal GetKey functions work for both
	master keys and subkeys.  */


static PGPError
pgpReturnPropBuffer (char const *src, void *prop, 
					 PGPSize srclen, PGPSize proplen)
{
	PGPError result = kPGPError_NoErr;

	if (srclen > proplen) {
		srclen = proplen;
		result = kPGPError_BufferTooSmall;
	}
	if ( IsntNull( prop ) && srclen > 0)
	{
		pgpCopyMemory( src, prop, srclen);
		/* Null terminate if there is room */
		if( srclen < proplen )
			*((PGPByte *)prop+srclen) = '\0';
	}
	return result;
}


/* Allocate an extra byte to leave room for null terminator in some cases */
static PGPError
pgpReturnAllocBuffer (PGPContextRef context, PGPByte const *databuf, 
					  PGPSize datalen, PGPByte **allocBuf)
{
	if( datalen == 0 )
		return kPGPError_InvalidProperty;
	*allocBuf = pgpContextMemAlloc( context, datalen+1, 0 );
	if( IsNull( *allocBuf ) )
		return kPGPError_OutOfMemory;
	pgpCopyMemory( databuf, *allocBuf, datalen );
	(*allocBuf)[datalen] = '\0';
	return kPGPError_NoErr;
}



	PGPError
pgpGetKeyNumberInternal (PGPKeyDBObj *key, PGPKeyDBObjProperty propname,
	PGPInt32 *prop)
{	
	PGPKeyDBRef			keys;
	PGPContextRef		context;
	PGPError            error = kPGPError_NoErr;
	PGPKeyDBObjRef		userid;
	PGPByte				pkalg;
	PGPCipherAlgorithm	lockAlg;
	PGPUInt32			lockBits;
	PGPBoolean			booldummy1, booldummy2;
	PGPEnv const *		pgpEnv;

	keys = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keys );
	pgpEnv = pgpContextGetEnvironment( context );

	pgpAssert( pgpObjectType( key ) == RINGTYPE_KEY ||
			   pgpObjectType( key ) == RINGTYPE_SUBKEY );

	switch (propname) {
	case kPGPKeyProperty_Validity:
		/* Use validity of primary userid */
		*prop = kPGPValidity_Unknown;
		userid = pgpKeyPrimaryUserID (key, 0);
		if( pgpKeyDBObjIsReal( userid ) )
			pgpGetUserIDNumber (userid, kPGPUserIDProperty_Validity, prop);
		break;
	case kPGPKeyProperty_Trust:
		*prop = pgpTrustExternToOld( pgpEnv,
									  pgpTrustToExtern(
										  pgpKeyConfidence( key ) ) );
		break;
	case kPGPKeyProperty_AlgorithmID:
		pgpKeyID8 (key, &pkalg, NULL);
		*prop = (long) pkalg;
		break;
	case kPGPKeyProperty_Bits:
		*prop = (long) pgpKeyBits (key);
		break;
	case kPGPKeyProperty_LockingAlgorithmID:
	case kPGPKeyProperty_LockingBits:
		if( !pgpKeyIsSec (key) )
			return kPGPError_InvalidProperty;
		if( IsPGPError( error = pgpSecProperties( key, &booldummy1,&booldummy2,
												  &lockAlg, &lockBits ) ) )
			break;
		if( propname == kPGPKeyProperty_LockingAlgorithmID )
			*prop = (long) lockAlg;
		else
			*prop = (long) lockBits;
		break;
	case kPGPKeyProperty_Flags:
	case kPGPKeyProperty_KeyServerPreferences:
		{
			PGPUInt32 flags = 0;
			PGPByte const *pflags;
			PGPSize flaglen, i;
			PGPBoolean hashed;
			PGPInt32 packettype;

			packettype = (propname == kPGPKeyProperty_Flags) ?
				SIGSUB_KEYFLAGS : SIGSUB_KEYSERVER_PREFERENCES;
			pflags = pgpKeyFindSubpacket( key, packettype, 0, &flaglen,
										  NULL, &hashed, NULL, NULL, &error );
			if( IsPGPError( error ) )
				return error;
			if( IsNull( pflags ) || !hashed )
				return kPGPError_InvalidProperty;

			/* Only return first n bytes */
			if( flaglen > sizeof(flags) )
				flaglen = sizeof(flags);

			/* Pack bytes into flags word, first into LSB */
			for( i=0; i<flaglen; ++i ) {
				flags |= pflags[i] << (i*8);
			}

			/* Return flags */
			*prop = flags;
			break;
		}
	case kPGPKeyProperty_HashAlgorithmID:
		if( pgpKeyV3( key ) )
			*prop = pgpenvGetInt(pgpEnv, PGPENV_HASH, NULL, NULL);
		else
			*prop = kPGPHashAlgorithm_SHA;
		break;
	case kPGPKeyProperty_Version:
		*prop = (PGPUInt32) pgpKeyVersion( key );
		break;
	case kPGPKeyProperty_TokenNum:
		*prop = sGetKeyTokenNum( key );
		break;
	default:
		return kPGPError_InvalidProperty;
	}
	return error;
}


	PGPError
pgpGetKeyNumber(PGPKeyDBObj *key, PGPKeyDBObjProperty propname, PGPInt32 *prop)
{
	PGPValidatePtr( prop );
	*prop	= 0;
	PGPValidateKeyOrSubKey( key );

	return pgpGetKeyNumberInternal( key, propname, prop );
}



	PGPError
pgpGetSubKeyNumberInternal (
	PGPKeyDBObjRef subkey, PGPKeyDBObjProperty propname, PGPInt32 *prop)
{
	pgpAssert( pgpObjectType( subkey ) == RINGTYPE_SUBKEY );

	switch (propname) {
	case kPGPSubKeyProperty_AlgorithmID:
		propname = kPGPKeyProperty_AlgorithmID;
		break;
	case kPGPSubKeyProperty_Bits:
		propname = kPGPKeyProperty_Bits;
		break;
	case kPGPSubKeyProperty_LockingAlgorithmID:
		propname = kPGPKeyProperty_LockingAlgorithmID;
		break;
	case kPGPSubKeyProperty_LockingBits:
		propname = kPGPKeyProperty_LockingBits;
		break;
	case kPGPSubKeyProperty_Version:
		propname = kPGPKeyProperty_Version;
		break;
	case kPGPSubKeyProperty_Flags:
		propname = kPGPKeyProperty_Flags;
		break;
	default:
		return kPGPError_InvalidProperty;
	}

	return pgpGetKeyNumberInternal (subkey, propname, prop );
}

	PGPError
pgpGetSubKeyNumber (
	PGPKeyDBObjRef subkey, PGPKeyDBObjProperty propname, PGPInt32 *prop)
{
	PGPValidatePtr( prop );
	*prop	= 0;
	PGPValidateSubKey( subkey );

	return pgpGetSubKeyNumberInternal( subkey, propname, prop );
}

static PGPError
pgpGetKeyTimeInternal (PGPKeyDBObj *key,
					   PGPKeyDBObjProperty propname, PGPTime *prop)
{
	PGPKeyDBObj const *crl;

	pgpAssert( pgpObjectType( key ) == RINGTYPE_KEY ||
			   pgpObjectType( key ) == RINGTYPE_SUBKEY );

	switch (propname) {
	case kPGPKeyProperty_Creation:
		*prop = pgpKeyCreation (key);
		break;
	case kPGPKeyProperty_Expiration:
		*prop = pgpKeyExpiration (key);
		break;
	case kPGPKeyProperty_CRLThisUpdate:
		crl = pgpKeyEarliestCRL( key, FALSE );
		if( IsNull( crl ) )
			return kPGPError_InvalidProperty;
		*prop = pgpCRLCreation( crl );
		break;
	case kPGPKeyProperty_CRLNextUpdate:
		crl = pgpKeyEarliestCRL( key, TRUE );
		if( IsNull( crl ) )
			return kPGPError_InvalidProperty;
		*prop = pgpCRLExpiration( crl );
		break;
	default:
		return kPGPError_InvalidProperty;
	}
	return kPGPError_NoErr;
}


	PGPError
pgpGetKeyTime (PGPKeyDBObj *key, PGPKeyDBObjProperty propname, PGPTime *prop)
{
	PGPValidatePtr( prop );
	*prop	= 0;
	PGPValidateKeyOrSubKey( key );

	return pgpGetKeyTimeInternal (key, propname, prop);
}


	PGPError
pgpGetSubKeyTime (PGPKeyDBObjRef subkey, PGPKeyDBObjProperty propname,
	PGPTime *prop)
{
	PGPValidatePtr( prop );
	*prop	= 0;
	PGPValidateSubKey( subkey );

	switch( propname ) {
	case kPGPSubKeyProperty_Creation:
		propname = kPGPKeyProperty_Creation;
		break;
	case kPGPSubKeyProperty_Expiration:
		propname = kPGPKeyProperty_Expiration;
		break;
	default:
		return kPGPError_InvalidProperty;
	}

	return pgpGetKeyTimeInternal (subkey, propname, prop);
}


	static PGPError
pgpGetKeyStringInternal(
	 PGPKeyDBObj *			key, 
	 PGPKeyDBObjProperty	propname,
	 void *					prop,
	 PGPSize				bufferSize,
	 PGPSize *				actualLength,
	 PGPBoolean				doAlloc,
	 PGPByte **				allocBuf)
{
	uchar                buffer[pgpMax(20, sizeof(PGPKeyID))];
	PGPCipherAlgorithm * prefAlgsLong;
	PGPByte const *		 prefData;
	PGPUInt32			 i;
	PGPKeyDBRef			 keys;
	PGPContextRef		 context;
	PGPBoolean			 hashed;
	PGPError			 err;

	pgpAssert( pgpObjectType( key ) == RINGTYPE_KEY ||
			   pgpObjectType( key ) == RINGTYPE_SUBKEY );

	keys = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keys );

	switch (propname) {
	default:
		return kPGPError_InvalidProperty;

	case kPGPKeyProperty_KeyID:
		pgpKeyID8 (key, NULL, (PGPKeyID *)&buffer);
		*actualLength = sizeof(PGPKeyID);
		if( doAlloc )
			return pgpReturnAllocBuffer( context, buffer,
										 *actualLength, allocBuf );
		break;

	case kPGPKeyProperty_Fingerprint:
		if (pgpKeyV3(key)) {
			pgpKeyFingerprint16 (key, buffer);
			*actualLength = 16;
		}
		else {
			pgpKeyFingerprint20 (key, buffer);
			*actualLength = 20;
		}
		if( doAlloc )
			return pgpReturnAllocBuffer( context, buffer,
										 *actualLength, allocBuf );
		break;

	case kPGPKeyProperty_ThirdPartyRevocationKeyID:
	{
		PGPKeyID		keyid;
		PGPByte const *	idBytes;

		if (!pgpKeyHasThirdPartyRevocation (key,
											NULL, NULL, &keyid, &err)) {
			return kPGPError_BadParams;
		}
		idBytes	= pgpGetKeyIDBytes( &keyid );
		for (i = 0; i < 4; i++)
			buffer[i] = idBytes[i+4];

		*actualLength = 4;
		if( doAlloc )
			return pgpReturnAllocBuffer( context, buffer,
										 *actualLength, allocBuf );
		break;
	}

	case kPGPKeyProperty_PreferredAlgorithms:
		/* Must convert from byte form to array of PGPCipherAlgorithm */
		prefData = pgpKeyFindSubpacket (
				key, SIGSUB_PREFERRED_ENCRYPTION_ALGS, 0,
				actualLength, NULL, &hashed, NULL, NULL, &err);
		if( IsNull( prefData ) || !hashed )
			return kPGPError_InvalidProperty;
		prefAlgsLong = (PGPCipherAlgorithm *)pgpContextMemAlloc ( context,
							*actualLength*sizeof(PGPCipherAlgorithm), 0 );
		if( IsNull( prefAlgsLong ) )
			return kPGPError_OutOfMemory;
		for (i=0; i < *actualLength; ++i) {
			prefAlgsLong[i] = (PGPCipherAlgorithm)prefData[i];
		}
		*actualLength *= sizeof(PGPCipherAlgorithm);
		if( doAlloc )
		{
			*allocBuf = (PGPByte *)prefAlgsLong;
			return kPGPError_NoErr;
		}
		err = pgpReturnPropBuffer (
				(char const *)prefAlgsLong, prop, *actualLength,
				bufferSize );
		pgpContextMemFree( context, prefAlgsLong );
		return err;

	case kPGPKeyProperty_PreferredKeyServer:
		prefData = pgpKeyFindSubpacket (
				key, SIGSUB_PREFERRED_KEYSERVER, 0,
				actualLength, NULL, &hashed, NULL, NULL, &err);
		if( IsNull( prefData ) || !hashed )
			return kPGPError_InvalidProperty;
		if( doAlloc )
			return pgpReturnAllocBuffer( context, prefData,
										 *actualLength, allocBuf );
		err = pgpReturnPropBuffer ((char const *)prefData, prop, *actualLength,
								   bufferSize );
		return err;

	case kPGPKeyProperty_KeyData:
		{	/* MPI data from key, algorithm specific */
			PGPByte const *keyData;
			PGPSize keyDataLength;

			keyData = pgpKeyMPIData( key, &keyDataLength );
			*actualLength = keyDataLength;
			if( doAlloc )
				return pgpReturnAllocBuffer( context, keyData,
											 *actualLength, allocBuf );
			err = pgpReturnPropBuffer ( (char *) keyData, prop, *actualLength,
										bufferSize);
			return err;
		}

	case kPGPKeyProperty_X509MD5Hash:
		{	/* Hash the key data in X.509 SubjPubKeyInfo (SPKI) format */
			PGPByte *keyData;
			PGPSize keyDataLength;
			PGPHashContextRef hc;

			if( IsNull( prop ) && !doAlloc ) {
				*actualLength = 16;
				return kPGPError_NoErr;
			}

			err = pgpKeyToX509SPKI( context, key, NULL,
									&keyDataLength );
			if (IsPGPError( err ))
				return err;
			keyData = (PGPByte *)pgpContextMemAlloc( context,
													 keyDataLength, 0);
			if( IsNull( keyData ) )
				return kPGPError_OutOfMemory;
			err = pgpKeyToX509SPKI( context, key, keyData,
									&keyDataLength );
			if (IsPGPError( err )) {
				pgpContextMemFree( context, keyData );
				return err;
			}

			err = PGPNewHashContext( context, kPGPHashAlgorithm_MD5, &hc );
			if( IsPGPError( err ) ) {
				pgpContextMemFree( context, keyData );
				return err;
			}
			PGPContinueHash( hc, keyData, keyDataLength );
			pgpAssert (sizeof(buffer) >= 16);
			PGPFinalizeHash( hc, buffer );
			PGPFreeHashContext( hc );
			*actualLength = 16;
			pgpContextMemFree( context, keyData );
			if( doAlloc )
				return pgpReturnAllocBuffer( context, buffer,
											 *actualLength, allocBuf );
			break;
		}
	}

	return pgpReturnPropBuffer ( (char const *)buffer,
			prop, *actualLength, bufferSize);
}

	PGPError
pgpGetKeyPropertyBuffer(
	PGPKeyDBObjRef		key,
	PGPKeyDBObjProperty	propname,
	PGPSize			bufferSize,
	void *			outData,
	PGPSize *		outLength )
{
	PGPError			err	= kPGPError_NoErr;

	PGPValidatePtr( outLength );
	*outLength	= 0;
	PGPValidateKeyOrSubKey( key );
	/* outData is allowed to be NULL */
	if ( IsntNull( outData ) )
	{
		pgpClearMemory( outData, bufferSize );
	}

	err	= pgpGetKeyStringInternal (key,
			propname, outData, bufferSize, outLength, FALSE, NULL );

	return( err );
}

	PGPError
pgpGetKeyAllocatedPropertyBuffer(
	PGPKeyDBObjRef		key,
	PGPKeyDBObjProperty	propname,
	PGPByte **			allocBuf,
	PGPSize *			outLength )
{
	PGPError			err	= kPGPError_NoErr;

	PGPValidatePtr( outLength );
	*outLength	= 0;
	PGPValidatePtr( allocBuf );
	*allocBuf	= NULL;
	PGPValidateKeyOrSubKey( key );

	err	= pgpGetKeyStringInternal (key,
			propname, NULL, 0, outLength, TRUE, allocBuf );

	return( err );
}


	PGPError
pgpGetSubKeyPropertyBuffer(
	PGPKeyDBObjRef		subKey,
	PGPKeyDBObjProperty	propname,
	PGPSize				bufferSize,
	void *				outData,
	PGPSize *			outLength )
{
	PGPError			err	= kPGPError_NoErr;

	PGPValidatePtr( outLength );
	*outLength	= 0;
	PGPValidateSubKey( subKey );
	if ( IsntNull( outData ) )
		pgpClearMemory( outData, bufferSize );

	CHECKREMOVED(subKey);

	switch (propname) {
	case kPGPSubKeyProperty_KeyData:
		propname = kPGPKeyProperty_KeyData;
		break;
	case kPGPSubKeyProperty_KeyID:
		propname = kPGPKeyProperty_KeyID;
		break;
	default:
		return kPGPError_InvalidProperty;
	}

	err	= pgpGetKeyStringInternal (subKey,
			propname, outData, bufferSize, outLength, FALSE, NULL );

	return( err );
}

	PGPError
pgpGetSubKeyAllocatedPropertyBuffer(
	PGPKeyDBObjRef		subKey,
	PGPKeyDBObjProperty	propname,
	PGPByte **			allocBuf,
	PGPSize *			outLength )
{
	PGPError			err	= kPGPError_NoErr;

	PGPValidatePtr( outLength );
	*outLength	= 0;
	PGPValidatePtr( allocBuf );
	*allocBuf	= NULL;
	PGPValidateSubKey( subKey );

	CHECKREMOVED(subKey);

	switch (propname) {
	case kPGPSubKeyProperty_KeyData:
		propname = kPGPKeyProperty_KeyData;
		break;
	case kPGPSubKeyProperty_KeyID:
		propname = kPGPKeyProperty_KeyID;
		break;
	default:
		return kPGPError_InvalidProperty;
	}

	err	= pgpGetKeyStringInternal (subKey,
			propname, NULL, 0, outLength, TRUE, allocBuf );

	return( err );
}

/* Check other aspects of key usability for sign/encrypt etc. */
static PGPError
sIsUsableKey (PGPKeyDBObj *key,
			  PGPKeyDBObjProperty propname, PGPBoolean *usable)
{
	PGPInt32		pkalg;
	PGPInt32		bits;
	PGPError		err = kPGPError_NoErr;

	pgpAssert (IsntNull( usable ) );
	(void) propname;

	*usable = TRUE;

	/* Disallow RSA keys bigger than BSafe supports */

	err = pgpGetKeyNumberInternal( key, kPGPKeyProperty_AlgorithmID,
								   &pkalg );
	if( IsPGPError( err) )
		return err;
	err = pgpGetKeyNumberInternal( key, kPGPKeyProperty_Bits,
								   &bits );
	if( IsPGPError( err) )
		return err;

	if( pkalg <= kPGPPublicKeyAlgorithm_RSA + 2 && bits > MAXRSABITS)
		*usable = FALSE;

	return err;
}


static PGPError
pgpGetKeyBooleanInternal (PGPKeyDBObj *key,
						  PGPKeyDBObjProperty propname, PGPBoolean *prop)
{
	PGPKeyDBRef			keys;
	PGPUInt32			expiration;
	PGPBoolean			needsPassphrase;
	PGPBoolean			isSecretShared;
	PGPCipherAlgorithm	cipherdummy;
	PGPUInt32			uintdummy;
	PGPError			err = kPGPError_NoErr;

	pgpAssert( pgpObjectType( key ) == RINGTYPE_KEY ||
			   pgpObjectType( key ) == RINGTYPE_SUBKEY );

	keys = PGPPeekKeyDBObjKeyDB( key );
	switch (propname) {
	case kPGPKeyProperty_IsSecret:
		*prop = (pgpKeyIsSec (key) != 0);
		break;
	case kPGPKeyProperty_IsAxiomatic:
		*prop = (pgpKeyAxiomatic (key) != 0);
		break;
	case kPGPKeyProperty_IsRevoked:
		*prop = (pgpKeyRevoked (key) != 0);
		/* "Or" in revocation status of master key on subkeys */
		if (!*prop &&
			pgpKeyIsSubkey (key))
			*prop = (pgpKeyRevoked (pgpKeyMasterkey(key))) != 0;
		break;
	case kPGPKeyProperty_IsRevocable:
		if (pgpKeyIsSec (key)) {
			*prop = TRUE;
		} else {
			PGPKeyDBObj *revobj;
			PGPByte revclass;
			PGPUInt32 i;

			*prop = FALSE;
			if (pgpKeyIsSubkey (key))
				key = pgpKeyMasterkey (key);
			pgpAssert (IsntNull( key ) );
			for (i=0; ; ++i) {
				revobj = pgpKeyRevocationKey (key, i, NULL, NULL,
											   &revclass, NULL, &err);
				if( IsPGPError( err ) ) {
					if( err == kPGPError_ItemNotFound )
						err = kPGPError_NoErr;
					break;
				}
				if( IsNull( revobj ) )
					continue;
				if ((revclass & 0x80) == 0)
					continue;
				if (pgpKeyIsSec (revobj)) {
					*prop = TRUE;
					break;
				}
			}
		}
		break;

	case kPGPKeyProperty_HasThirdPartyRevocation:
		*prop = pgpKeyHasThirdPartyRevocation (key, NULL, NULL, NULL, &err);
		break;

	case kPGPKeyProperty_NeedsPassphrase:
	case kPGPKeyProperty_IsSecretShared:
		if (!pgpKeyIsSec (key))
		{
			*prop = FALSE;
			break;
		}
		if( IsPGPError( err = pgpSecProperties( key, &needsPassphrase,
												&isSecretShared,
												&cipherdummy, &uintdummy ) ) )
			break;
		if( propname == kPGPKeyProperty_NeedsPassphrase )
			*prop = needsPassphrase;
		else
			*prop = isSecretShared;
		break;
	case kPGPKeyProperty_HasUnverifiedRevocation:
		{
			/*
			 * True if there is a revocation signature.
			 * Not checking key signer because since we are not verifying,
			 * it would not add meaningful security.  This property can only
			 * be considered advisory, as its name implies.
			 */
			PGPKeyID			keyid;
			PGPKeyDBObj *		child;

			*prop = FALSE;
			pgpKeyID8 (key, NULL, &keyid);
			for( child = key->down; IsntNull( child ); child = child->next )
			{
				if (!pgpKeyDBObjIsReal( child ) )
					continue;
				if (pgpObjectType(child) == RINGTYPE_SIG &&
					pgpSigType (child) == PGP_SIGTYPE_KEY_REVOKE) {
					*prop = TRUE;
					break;
				}
			}
		}
		break;
	case kPGPKeyProperty_IsDisabled:
		*prop = (pgpKeyDisabled (key) != 0);
		break;
	case kPGPKeyProperty_IsExpired:
		expiration = pgpKeyExpiration (key);
		if (expiration == 0)
			*prop = 0;
		else
			*prop = (expiration < (PGPUInt32) PGPGetTime());
		break;
	case kPGPKeyProperty_IsOnToken:
		*prop = pgpKeyIsOnToken( key );
		break;
	case kPGPKeyProperty_IsNotCorrupt:
		*prop = TRUE;
		break;
	case kPGPKeyProperty_IsSigningKey:
		*prop = ((pgpKeyUse (key) & PGP_PKUSE_SIGN) != 0);
		break;
	case kPGPKeyProperty_IsEncryptionKey:
		*prop = ((pgpKeyUseRevokedOK (key) & PGP_PKUSE_ENCRYPT) != 0);
		break;

	case kPGPKeyProperty_CanEncrypt:
	case kPGPKeyProperty_CanSign:
	{
		/* Not corrupted and not revoked and not expired and not disabled */

		PGPBoolean	notCorrupted;

		*prop = FALSE;

		if( propname == kPGPKeyProperty_CanSign )
		{
			PGPBoolean	isSecretKey;

			/* Quick reject non-secret keys in the signing case */
			err = pgpGetKeyBooleanInternal( key,
						kPGPKeyProperty_IsSecret, &isSecretKey );
			if( IsPGPError( err ) || ! isSecretKey )
				break;
		}

		err = pgpGetKeyBooleanInternal( key,
						kPGPKeyProperty_IsNotCorrupt, &notCorrupted );
		if( IsntPGPError( err ) && notCorrupted )
		{
			PGPBoolean	isRevoked;

			err = pgpGetKeyBooleanInternal( key,
							kPGPKeyProperty_IsRevoked, &isRevoked );
			if( IsntPGPError( err ) && ! isRevoked )
			{
				PGPBoolean	isExpired;

				err = pgpGetKeyBooleanInternal( key,
							kPGPKeyProperty_IsExpired, &isExpired );
				if( IsntPGPError( err ) && ! isExpired )
				{
					PGPBoolean	isDisabled;

					err = pgpGetKeyBooleanInternal( key,
								kPGPKeyProperty_IsDisabled, &isDisabled );
					if( IsntPGPError( err ) && ! isDisabled )
					{
						PGPBoolean isUsable;

						err = sIsUsableKey( key, propname,
											&isUsable );
						if( IsntPGPError( err ) && isUsable )
						{
							if( propname == kPGPKeyProperty_CanEncrypt )
							{
								*prop = ((pgpKeyUnexpiredUse(key)
										  & PGP_PKUSE_ENCRYPT) != 0);
							}
							else
							{
								err = pgpGetKeyBooleanInternal(
									key, kPGPKeyProperty_IsSigningKey, prop );
							}
						}
					}
				}
			}
		}

		break;
	}

	case kPGPKeyProperty_CanDecrypt:
	{
		PGPBoolean	isSecretKey;

		/* Is secret key and not corrupted and is encryption key */

		*prop = FALSE;

		err = pgpGetKeyBooleanInternal( key,
					kPGPKeyProperty_IsSecret, &isSecretKey );
		if( IsntPGPError( err ) && isSecretKey )
		{
			PGPBoolean	notCorrupted;

			err = pgpGetKeyBooleanInternal( key,
							kPGPKeyProperty_IsNotCorrupt, &notCorrupted );
			if( IsntPGPError( err ) && notCorrupted )
			{
				PGPBoolean isUsable;

				err = sIsUsableKey( key, propname,
									&isUsable );
				if( IsntPGPError( err ) && isUsable )
				{
					err = pgpGetKeyBooleanInternal( key,
									kPGPKeyProperty_IsEncryptionKey, prop );
				}
			}
		}

		break;
	}

	case kPGPKeyProperty_CanVerify:
	{
		/* Can verify if not corrupted and a signature key */
		PGPBoolean	notCorrupted;

		err = pgpGetKeyBooleanInternal( key,
						kPGPKeyProperty_IsNotCorrupt, &notCorrupted );
		if( IsntPGPError( err ) && notCorrupted )
		{
			PGPBoolean isUsable;

			err = sIsUsableKey( key, propname,
								&isUsable );
			if( IsntPGPError( err ) && isUsable )
			{
				err = pgpGetKeyBooleanInternal( key,
								kPGPKeyProperty_IsSigningKey, prop );
			}
		}
		break;
	}

	case kPGPKeyProperty_HasCRL:
		*prop = pgpKeyHasCRL(key );
		break;

	default:
		err = kPGPError_InvalidProperty;
		break;
	}

	return err;
}


	PGPError
pgpGetKeyBoolean (PGPKeyDBObj *key, PGPKeyDBObjProperty propname,
	PGPBoolean *prop)
{
	PGPValidatePtr( prop );
	*prop	= FALSE;
	PGPValidateKeyOrSubKey( key );

	return pgpGetKeyBooleanInternal (key, propname, prop);
}


	PGPError
pgpGetSubKeyBoolean (
	PGPKeyDBObjRef subkey, PGPKeyDBObjProperty propname, PGPBoolean *prop)
{
	pgpa(pgpaPGPSubKeyValid(subkey));

	PGPValidatePtr( prop );
	*prop	= FALSE;
	PGPValidateSubKey( subkey );

	switch( propname ) {
	case kPGPSubKeyProperty_IsRevoked:
		propname = kPGPKeyProperty_IsRevoked;
		break;
	case kPGPSubKeyProperty_IsNotCorrupt:
		propname = kPGPKeyProperty_IsNotCorrupt;
		break;
	case kPGPSubKeyProperty_IsExpired:
		propname = kPGPKeyProperty_IsExpired;
		break;
	case kPGPSubKeyProperty_IsOnToken:
		propname = kPGPKeyProperty_IsOnToken;
		break;
	case kPGPSubKeyProperty_NeedsPassphrase:
		propname = kPGPKeyProperty_NeedsPassphrase;
		break;
	case kPGPSubKeyProperty_HasUnverifiedRevocation:
		propname = kPGPKeyProperty_HasUnverifiedRevocation;
		break;
	case kPGPSubKeyProperty_IsRevocable:
		propname = kPGPKeyProperty_IsRevocable;
		break;
	case kPGPSubKeyProperty_HasThirdPartyRevocation:
		propname = kPGPKeyProperty_HasThirdPartyRevocation;
		break;
	default:
		return kPGPError_InvalidProperty;
	}

	return pgpGetKeyBooleanInternal (subkey, propname, prop);
}


	PGPError
pgpGetUserIDNumberInternal (
	PGPKeyDBObj *userid, PGPKeyDBObjProperty propname, PGPInt32 *prop)
{
	PGPKeyDBRef			keys;
	PGPContextRef		context;
	PGPEnv *			pgpEnv;

	keys = PGPPeekKeyDBObjKeyDB( userid );
	context = PGPPeekKeyDBContext( keys );
	pgpEnv = pgpContextGetEnvironment( context );

	switch (propname) {
	case kPGPUserIDProperty_Validity:
		*prop = (PGPInt32)pgpUserIDOldValidity( userid );
		break;
	case kPGPUserIDProperty_Confidence: 
		*prop = (PGPInt32)pgpUserIDConfidence (userid);
		break;
	case kPGPUserIDProperty_AttributeType:
		(void) pgpUserIDAttributeSubpacket( userid, 0,
										   (PGPUInt32 *)prop, NULL, NULL );
		break;

	default:
		return kPGPError_InvalidProperty;
	}

	return kPGPError_NoErr;
}

	PGPError
pgpGetUserIDNumber (
	PGPKeyDBObj *userid, PGPKeyDBObjProperty propname, PGPInt32 *prop)
{
	pgpa(pgpaPGPUserIDValid(userid));

	PGPValidatePtr( prop );
	*prop	= 0;
	PGPValidateUserID( userid );

	CHECKREMOVED(userid);

	return pgpGetUserIDNumberInternal( userid, propname, prop );
}

	PGPError
pgpGetUserIDBooleanInternal (
	PGPKeyDBObj *userid, PGPKeyDBObjProperty propname, PGPBoolean *prop)
{

	switch (propname) {
	case kPGPUserIDProperty_IsAttribute:
		*prop = pgpUserIDIsAttribute( userid ) != 0;
		break;
	default:
		return kPGPError_InvalidProperty;
	}
	return kPGPError_NoErr;
}


	PGPError
pgpGetUserIDBoolean (
	PGPKeyDBObj *userid, PGPKeyDBObjProperty propname, PGPBoolean *prop)
{
	pgpa(pgpaPGPUserIDValid(userid));

	PGPValidatePtr( prop );
	*prop	= FALSE;
	PGPValidateUserID( userid );

	CHECKREMOVED(userid);

	return pgpGetUserIDBooleanInternal( userid, propname, prop );
}


/*____________________________________________________________________________
	Name is always returned NULL terminated.

	if 'outString' is NULL, then just the size is returned.
____________________________________________________________________________*/
	static PGPError
pgpGetUserIDStringBufferInternal(
	PGPKeyDBObjRef		userid,
	PGPKeyDBObjProperty	propname,
	PGPSize				bufferSize,
	char *				outString,
	PGPSize *			fullLengthOut,
	PGPBoolean			doAlloc,
	PGPByte **			allocBuf)
{
	PGPKeyDBRef			keys;
	PGPContextRef		context;
	char const	        *bufptr, *bufptr2;
	PGPError			err	= kPGPError_NoErr;
	PGPSize				fullLength;

	keys = PGPPeekKeyDBObjKeyDB( userid );
	context = PGPPeekKeyDBContext( keys );

	switch( propname ) {
	case kPGPUserIDProperty_Name:
		if (pgpUserIDIsAttribute (userid))
			return kPGPError_InvalidProperty;
		bufptr	= pgpUserIDName (userid, &fullLength );
		if( doAlloc )
		{
			err = pgpReturnAllocBuffer( context, (PGPByte *)bufptr,
										 fullLength, allocBuf );
			goto done;
		}
		break;

	case kPGPUserIDProperty_EmailAddress:
		if (pgpUserIDIsAttribute (userid))
			return kPGPError_InvalidProperty;
		bufptr2	= pgpUserIDName (userid, &fullLength );
		bufptr = (char *)memchr(bufptr2, '<', fullLength);
		if (bufptr == NULL)
			return kPGPError_InvalidProperty;
		bufptr++;
		fullLength -= (bufptr - bufptr2);
		bufptr2 = (char *)memchr(bufptr, '>', fullLength);
		if (bufptr2 != NULL)
			fullLength = bufptr2 - bufptr;
		if( doAlloc )
		{
			err = pgpReturnAllocBuffer( context, (PGPByte *)bufptr,
										 fullLength, allocBuf );
			goto done;
		}
		break;

	case kPGPUserIDProperty_CommonName:
		if (pgpUserIDIsAttribute (userid))
			return kPGPError_InvalidProperty;
		bufptr	= pgpUserIDName (userid, &fullLength );
		bufptr2 = (char *)memchr(bufptr, '<', fullLength);
		if (bufptr2 != NULL)
		{
			while (bufptr2 > bufptr && bufptr2[-1] == ' ')
				bufptr2--;
			fullLength = bufptr2 - bufptr;
		}
		if( doAlloc )
		{
			err = pgpReturnAllocBuffer( context, (PGPByte *)bufptr,
										 fullLength, allocBuf );
			goto done;
		}
		break;

	case kPGPUserIDProperty_AttributeData:
		if (!pgpUserIDIsAttribute (userid))
			return kPGPError_InvalidProperty;
		bufptr	= (const char *)pgpUserIDAttributeSubpacket (userid,
									0, NULL, &fullLength, &err );
		if( IsPGPError( err ) )
			return err;
		if( doAlloc )
		{
			err = pgpReturnAllocBuffer( context, (PGPByte *)bufptr,
										 fullLength, allocBuf );
			goto done;
		}
		break;

	default:
		return kPGPError_InvalidProperty;
	}

	err	= pgpReturnPropBuffer ( bufptr,
		(PGPByte *)outString, fullLength, bufferSize);

 done:
	if ( IsntNull( fullLengthOut ) )
		*fullLengthOut	= fullLength;

	return( err );
}


	PGPError
pgpGetUserIDStringBuffer(
	PGPKeyDBObjRef		userid,
	PGPKeyDBObjProperty	propname,
	PGPSize				bufferSize,
	char *				outString,
	PGPSize *			fullLengthOut )
{

	if ( IsntNull( fullLengthOut ) )
		*fullLengthOut	= 0;
	PGPValidateUserID( userid );
#if PGP_DEBUG
	if ( IsntNull( outString ) )
		pgpClearMemory( outString, bufferSize );
#endif

	CHECKREMOVED(userid);

	return pgpGetUserIDStringBufferInternal( userid, propname,
						bufferSize, outString, fullLengthOut, FALSE, NULL );
}


	PGPError
pgpGetUserIDAllocatedStringBuffer(
	PGPKeyDBObjRef		userid,
	PGPKeyDBObjProperty	propname,
	PGPByte **			allocBuf,
	PGPSize *			outLength )
{

	if ( IsntNull( outLength ) )
		*outLength	= 0;
	if ( IsntNull( allocBuf ) )
		*allocBuf	= NULL;
	PGPValidateUserID( userid );

	CHECKREMOVED(userid);

	return pgpGetUserIDStringBufferInternal( userid, propname,
						0, NULL, outLength, TRUE, allocBuf );
}



	PGPError
pgpGetSigNumberInternal (PGPKeyDBObj *sig, PGPKeyDBObjProperty propname,
	PGPInt32 *prop)
{
	PGPByte				pkalg;

	switch (propname) {
	case kPGPSigProperty_AlgorithmID:
		pgpSigID8 (sig, &pkalg, NULL);
		*prop = (long) pkalg;
		break;
	case kPGPSigProperty_TrustLevel:
		*prop = (long)pgpSigTrustLevel(sig);
		break;
	case kPGPSigProperty_TrustValue:
		*prop = (long)pgpSigTrustValue(sig);
		break;
	default:
		return kPGPError_InvalidProperty;
	}
	return kPGPError_NoErr;

}

	PGPError
pgpGetSigNumber (PGPKeyDBObj *sig, PGPKeyDBObjProperty propname,
	PGPInt32 *prop)
{
	PGPValidatePtr( prop );
	*prop	= 0;
	PGPValidateSig( sig );
	
	CHECKREMOVED(sig);

	return pgpGetSigNumberInternal( sig, propname, prop );
}


	static PGPError
pgpGetSigStringInternal(
	PGPKeyDBObj *		sig, 
	PGPKeyDBObjProperty		propname,
	void *				prop,
	PGPSize				bufferSize,
	PGPSize *			actualLength,
	PGPBoolean			doAlloc,
	PGPByte **			allocBuf)
{
	PGPKeyDBRef			keys;
	PGPContextRef		context;
	PGPByte *			ptr;
	PGPSize				len;
	PGPKeyID			keyid;
	PGPError			error;

	keys = PGPPeekKeyDBObjKeyDB( sig );
	context = PGPPeekKeyDBContext( keys );

	switch (propname) {
	default:
		return kPGPError_InvalidProperty;
		
	case kPGPSigProperty_KeyID:
		/* Don't return the pseudo keyids we use for X.509 certs w/no signer */
		if( pgpKeyDBObjIsX509Dummy( pgpSigMakerDummyOK( sig ) ) )
			return kPGPError_InvalidProperty;

		pgpSigID8 (sig, NULL, &keyid);
		*actualLength = sizeof(PGPKeyID);
		ptr = (PGPByte *)&keyid;
		if( doAlloc )
			return pgpReturnAllocBuffer( context, ptr,
										 *actualLength, allocBuf );
		break;
	
	case kPGPSigProperty_X509Certificate:
		if (!pgpSigIsX509 (sig))
			return kPGPError_InvalidProperty;
		ptr = pgpSigX509Certificate(sig, &len);
		if (IsNull( ptr ) )
			return kPGPError_InvalidProperty;
		*actualLength = len;
		if( doAlloc )
			return pgpReturnAllocBuffer( context, ptr,
										 *actualLength, allocBuf );
		break;

	/* Return IssuerAndSerialNumber sequence for specified signature */
	case kPGPSigProperty_X509IASN:
		{	PGPByte *iasn;
			PGPSize iasnLength;

			if (!pgpSigIsX509 (sig))
				return kPGPError_InvalidProperty;
			ptr = pgpSigX509Certificate(sig, &len);
			if (IsNull( ptr ) )
				return kPGPError_InvalidProperty;
			error = pgpX509CertToIASN( ptr, len, NULL, &iasnLength);
			if (IsPGPError( error ))
				return error;
			*actualLength = iasnLength;
			if( doAlloc || IsntNull( prop ) ) {
				iasn = (PGPByte *)pgpContextMemAlloc( context, iasnLength, 0);
				if( IsNull( iasn ) )
					return kPGPError_OutOfMemory;
				error = pgpX509CertToIASN( ptr, len, iasn, NULL);
				if (IsPGPError( error ))
					return error;
				if( doAlloc )
				{
					*allocBuf = iasn;
					return kPGPError_NoErr;
				}
				if (bufferSize < iasnLength)
					iasnLength = bufferSize;
				pgpCopyMemory( iasn, prop, iasnLength );
				pgpContextMemFree( context, iasn );
			}
			return kPGPError_NoErr;
		}

	/* Return LongName for specified signature */
	case kPGPSigProperty_X509LongName:
	case kPGPSigProperty_X509IssuerLongName:
		{	PGPByte *name;
			PGPSize nameLength;
			PGPBoolean doIssuer = (propname==kPGPSigProperty_X509IssuerLongName);

			if (!pgpSigIsX509 (sig))
				return kPGPError_InvalidProperty;
			ptr = pgpSigX509Certificate(sig, &len);
			if (IsNull( ptr ) )
				return kPGPError_InvalidProperty;
			error = pgpX509CertToLongName( ptr, len, doIssuer,
										   NULL, &nameLength);
			if (IsPGPError( error ))
				return error;
			*actualLength = nameLength;
			if( doAlloc || IsntNull( prop ) ) {
				name = (PGPByte *)pgpContextMemAlloc( context, nameLength+1,0);
				if( IsNull( name ) )
					return kPGPError_OutOfMemory;
				error = pgpX509CertToLongName( ptr, len, doIssuer,
											   name, &nameLength);
				if (IsPGPError( error ))
					return error;
				if( doAlloc )
				{
					name[nameLength] = '\0';
					*allocBuf = name;
					return kPGPError_NoErr;
				}
				if (bufferSize < nameLength)
					nameLength = bufferSize;
				pgpCopyMemory( name, prop, nameLength );
				pgpContextMemFree( context, name );
			}
			return kPGPError_NoErr;
		}
	
	/* Return Subject DN for specified signature certificate */
	case kPGPSigProperty_X509DERDName:
		{	PGPByte *name;
			PGPSize nameLength;

			if (!pgpSigIsX509 (sig))
				return kPGPError_InvalidProperty;
			ptr = pgpSigX509Certificate(sig, &len);
			if (IsNull( ptr ) )
				return kPGPError_InvalidProperty;
			error = pgpX509CertToDName( ptr, len, FALSE, NULL, &nameLength);
			if (IsPGPError( error ))
				return error;
			*actualLength = nameLength;
			if( doAlloc || IsntNull( prop ) ) {
				name = (PGPByte *)pgpContextMemAlloc( context, nameLength,0);
				if( IsNull( name ) )
					return kPGPError_OutOfMemory;
				error = pgpX509CertToDName( ptr, len, FALSE,
											name, &nameLength);
				if (IsPGPError( error ))
					return error;
				if( doAlloc )
				{
					*allocBuf = name;
					return kPGPError_NoErr;
				}
				if (bufferSize < nameLength)
					nameLength = bufferSize;
				pgpCopyMemory( name, prop, nameLength );
				pgpContextMemFree( context, name );
			}
			return kPGPError_NoErr;
		}
	
	/* Return IP or DNS addr for specified signature */
	case kPGPSigProperty_X509IPAddress:
	case kPGPSigProperty_X509DNSName:
		{	PGPByte *val;
			PGPSize valLength;
			PGPBoolean doIP = (propname==kPGPSigProperty_X509IPAddress);

			if (!pgpSigIsX509 (sig))
				return kPGPError_InvalidProperty;
			ptr = pgpSigX509Certificate(sig, &len);
			if (IsNull( ptr ) )
				return kPGPError_InvalidProperty;
			error = pgpX509CertToIPDNS( ptr, len, doIP, NULL, &valLength);
			if (IsPGPError( error ))
				return error;
			*actualLength = valLength;
			/* Return error if there is no such data */
			if( valLength == 0 )
				return kPGPError_BadParams;
			if( doAlloc || IsntNull( prop ) ) {
				val = (PGPByte *)pgpContextMemAlloc( context, valLength+1, 0);
				if( IsNull( val ) )
					return kPGPError_OutOfMemory;
				error = pgpX509CertToIPDNS( ptr, len, doIP, val, &valLength);
				if (IsPGPError( error ))
					return error;
				if( doAlloc )
				{
					val[valLength] = '\0';
					*allocBuf = val;
					return kPGPError_NoErr;
				}
				if (bufferSize < valLength)
					valLength = bufferSize;
				pgpCopyMemory( val, prop, valLength );
				pgpContextMemFree( context, val );
			}
			return kPGPError_NoErr;
		}
	}
	
	return pgpReturnPropBuffer ( (char const *)ptr,
			prop, *actualLength, bufferSize);
}


	PGPError
pgpGetSigPropertyBuffer(
	PGPKeyDBObjRef		sig,
	PGPKeyDBObjProperty	propname,
	PGPSize			bufferSize,
	void *			outData,
	PGPSize *		outLength )
{
	PGPKeyDBRef			keys;
    PGPError			err	= kPGPError_NoErr;

	PGPValidatePtr( outLength );
	*outLength	= 0;
	PGPValidateSig( sig );
	/* outData is allowed to be NULL */
	if ( IsntNull( outData ) )
	{
		pgpClearMemory( outData, bufferSize );
	}
	
	keys = PGPPeekKeyDBObjKeyDB( sig );
	
	err	= pgpGetSigStringInternal( sig, propname, outData,
								   bufferSize, outLength, FALSE, NULL );

	return( err );
}

	PGPError
pgpGetSigAllocatedPropertyBuffer(
	PGPKeyDBObjRef		sig,
	PGPKeyDBObjProperty	propname,
	PGPByte **			allocBuf,
	PGPSize *			outLength )
{
	PGPKeyDBRef			keys;
    PGPError			err	= kPGPError_NoErr;

	PGPValidatePtr( allocBuf );
	PGPValidatePtr( outLength );
	*allocBuf = NULL;
	*outLength = 0;
	PGPValidateSig( sig );
	
	keys = PGPPeekKeyDBObjKeyDB( sig );
	
	err	= pgpGetSigStringInternal( sig, propname, NULL, 0, outLength,
								   TRUE, allocBuf );

	return( err );
}


	PGPError
pgpGetKeyIDOfCertifier(
	PGPKeyDBObj *		sig,
	PGPKeyID *			outID )
{
	PGPError			err	= kPGPError_NoErr;

	PGPValidatePtr( outID );
	pgpClearMemory( outID, sizeof( *outID ) );
	PGPValidateSig( sig );

	CHECKREMOVED(sig);

	pgpSigID8( sig, NULL, outID );
	return err;
}


	PGPError
pgpGetSigTimeInternal (PGPKeyDBObj *sig, PGPKeyDBObjProperty propname,
	PGPTime *prop)
{
	switch (propname) {
	case kPGPSigProperty_Creation:
		*prop = pgpSigTimestamp (sig);
		break;
	case kPGPSigProperty_Expiration:
		*prop = pgpSigExpiration (sig);
		break;
	default:
		return kPGPError_InvalidProperty;
	}
	return kPGPError_NoErr;
}


	PGPError
pgpGetSigTime (PGPKeyDBObj *sig, PGPKeyDBObjProperty propname, PGPTime *prop)
{
	PGPValidatePtr( prop );
	*prop	= 0;
	PGPValidateSig( sig );
	
	CHECKREMOVED(sig);

	return pgpGetSigTimeInternal( sig, propname, prop );
}




	PGPError
pgpGetSigBooleanInternal (PGPKeyDBObj *sig, PGPKeyDBObjProperty propname,
	PGPBoolean *prop)
{
	PGPKeyDBObj		    *sib = NULL;
	PGPKeyDBObj		    *obj;
	PGPKeyID			 keyid, revkeyid;
	PGPTime				 expiration;
	
	switch (propname) {
	case kPGPSigProperty_HasUnverifiedRevocation:
		/* True automatically if sig is revoked */
		if( pgpSigRevoked (sig) ) {
			*prop = 1;
			break;
		}
		/*  Must look for a revocation signature with the same signing 
			key id.  The revocation sig must be the newer than the certifying
			sig to be considered. */
		*prop = 0;
		pgpSigID8 (sig, NULL, &keyid);
		for( sib = sig->up->down; IsntNull(sib); sib = sib->next )
		{
			if (!pgpKeyDBObjIsReal( sib ) )
				continue;
			if (!OBJISSIG( sib ) )
				continue;
			if (pgpSigType (sib) == PGP_SIGTYPE_KEY_UID_REVOKE) {
				pgpSigID8 (sib, NULL, &revkeyid);
				if (pgpKeyIDsEqual( &keyid, &revkeyid ) &&
					       pgpSigTimestamp (sib) >= 
					           pgpSigTimestamp (sig)) {
					*prop = 1;
					break;
				}
			}
		}
		break;
	case kPGPSigProperty_IsRevoked:
		*prop = pgpSigRevoked (sig);
		break;
	case kPGPSigProperty_IsNotCorrupt:
		*prop = TRUE;
		break;
	case kPGPSigProperty_IsTried:
		*prop = pgpSigTried (sig);
		break;
	case kPGPSigProperty_IsVerified:
		*prop = pgpSigChecked (sig);
		break;
	case kPGPSigProperty_IsMySig:
		obj = pgpSigMaker (sig);
		if (!obj)
			*prop = 0;
		else
			*prop = pgpKeyIsSec (obj);
		break;
	case kPGPSigProperty_IsExportable:
		*prop = pgpSigExportable (sig);
		break;
	case kPGPSigProperty_IsExpired:
		expiration = pgpSigExpiration (sig);
		if (expiration == 0)
			*prop = 0;
		else
			*prop = (expiration < (PGPUInt32) PGPGetTime());
		break;
	case kPGPSigProperty_IsX509:
		*prop = pgpSigIsX509 (sig);
		break;
	default:
		return kPGPError_InvalidProperty;
	}
	return kPGPError_NoErr;
}


	PGPError
pgpGetSigBoolean (PGPKeyDBObj *sig, PGPKeyDBObjProperty propname,
	PGPBoolean *prop)
{
	PGPValidatePtr( prop );
	*prop	= FALSE;
	PGPValidateSig( sig );
	
	CHECKREMOVED(sig);

	return pgpGetSigBooleanInternal( sig, propname, prop );
}


#define kPGPKeyDBObjType_Mask				0xf

	PGPError
pgpGetKeyDBObjNumber( PGPKeyDBObj *obj, PGPKeyDBObjProperty propname,
	PGPInt32 *prop )
{
	PGPValidatePtr( prop );
	*prop	= 0;
	PGPValidateKeyDBObj( obj );
	switch( propname ) {

	default:
		return kPGPError_InvalidProperty;

	case kPGPKeyDBObjProperty_ObjectType:
		*prop = obj->objflags & kPGPKeyDBObjType_Mask;
		break;
	}

	return kPGPError_NoErr;
}


/* Generic key property functions */

#define kPGPKeyProperty_FirstProperty		100
#define kPGPSubKeyProperty_FirstProperty	500
#define kPGPUserIDProperty_FirstProperty	900
#define kPGPSigProperty_FirstProperty		1300


	PGPError
PGPGetKeyDBObjBooleanProperty( PGPKeyDBObjRef obj,
	PGPKeyDBObjProperty prop, PGPBoolean *rslt )
{
	pgpEnterPGPErrorFunction();

	if( prop < kPGPSubKeyProperty_FirstProperty )
		return pgpGetKeyBoolean( obj, prop, rslt );
	else if( prop < kPGPUserIDProperty_FirstProperty )
		return pgpGetSubKeyBoolean( obj, prop, rslt );
	else if( prop < kPGPSigProperty_FirstProperty )
		return pgpGetUserIDBoolean( obj, prop, rslt );
	else
		return pgpGetSigBoolean( obj, prop, rslt );
}

	PGPError
PGPGetKeyDBObjNumericProperty( PGPKeyDBObjRef obj,
	PGPKeyDBObjProperty prop, PGPInt32 *rslt )
{
	pgpEnterPGPErrorFunction();

	if( prop < kPGPKeyProperty_FirstProperty )
		return pgpGetKeyDBObjNumber( obj, prop, rslt );
	if( prop < kPGPSubKeyProperty_FirstProperty )
		return pgpGetKeyNumber( obj, prop, rslt );
	else if( prop < kPGPUserIDProperty_FirstProperty )
		return pgpGetSubKeyNumber( obj, prop, rslt );
	else if( prop < kPGPSigProperty_FirstProperty )
		return pgpGetUserIDNumber( obj, prop, rslt );
	else
		return pgpGetSigNumber( obj, prop, rslt );
}

	PGPError
PGPGetKeyDBObjTimeProperty( PGPKeyDBObjRef obj,
	PGPKeyDBObjProperty prop, PGPTime *rslt )
{
	pgpEnterPGPErrorFunction();

	if( prop < kPGPSubKeyProperty_FirstProperty )
		return pgpGetKeyTime( obj, prop, rslt );
	else if( prop < kPGPUserIDProperty_FirstProperty )
		return pgpGetSubKeyTime( obj, prop, rslt );
	else if( prop < kPGPSigProperty_FirstProperty )
		return kPGPError_BadParams;
	else
		return pgpGetSigTime( obj, prop, rslt );
}


	PGPError
PGPGetKeyDBObjDataProperty( PGPKeyDBObjRef obj,
	PGPKeyDBObjProperty prop, void *buffer, PGPSize bufSize, PGPSize *datSize)
{
	pgpEnterPGPErrorFunction();

	if( prop < kPGPSubKeyProperty_FirstProperty )
		return pgpGetKeyPropertyBuffer( obj, prop, bufSize, buffer, datSize );
	else if( prop < kPGPUserIDProperty_FirstProperty )
		return pgpGetSubKeyPropertyBuffer(obj, prop, bufSize, buffer, datSize);
	else if( prop < kPGPSigProperty_FirstProperty )
		return pgpGetUserIDStringBuffer( obj, prop, bufSize, buffer, datSize);
	else
		return pgpGetSigPropertyBuffer( obj, prop, bufSize, buffer, datSize );
}


	PGPError
PGPGetKeyDBObjAllocatedDataProperty( PGPKeyDBObjRef obj,
	PGPKeyDBObjProperty prop, void **buffer, PGPSize *dataSize)
{
	PGPError err;
	PGPByte *buf;

	pgpEnterPGPErrorFunction();

	if( prop < kPGPSubKeyProperty_FirstProperty )
		err = pgpGetKeyAllocatedPropertyBuffer( obj, prop, &buf, dataSize );
	else if( prop < kPGPUserIDProperty_FirstProperty )
		err = pgpGetSubKeyAllocatedPropertyBuffer(obj, prop, &buf, dataSize);
	else if( prop < kPGPSigProperty_FirstProperty )
		err = pgpGetUserIDAllocatedStringBuffer( obj, prop, &buf, dataSize);
	else
		err = pgpGetSigAllocatedPropertyBuffer( obj, prop, &buf, dataSize );

	*buffer = (void *)buf;
	return err;
}


#if 0
/*  Get and Set default private key.  The identification of
	the key is stored as an ascii keyid in the preferences
	repository. */


	PGPError 
PGPGetDefaultPrivateKey (
	PGPKeyDBRef			keydb,
	PGPKeyDBObjRef *	outRef )
{
	PGPError	err	= kPGPError_NoErr;
	
	PGPValidatePtr( outRef );
	*outRef	= NULL;
	PGPValidateKeyDB( keydb );
	
	pgpEnterPGPErrorFunction();

	err	= pgpGetDefaultPrivateKeyInternal( keyDB, outRef );
	
	pgpAssertErrWithPtr( err, *outRef );
	return( err );
}



	PGPError
PGPSetDefaultPrivateKey (PGPKeyDBObjRef	key)
{
	PGPKeyDBRef		keydb;
	PGPContextRef	context;
	PGPBoolean		isSecret = 0;
	PGPBoolean		cansign = 0;
	PGPError		err	= kPGPError_NoErr;
	PGPKeyID		keyID;
	
	PGPValidateKey( key );
	
	pgpEnterPGPErrorFunction();

	err	= pgpKeyDeadCheck( key) ;
	if ( IsPGPError( err ) )
		return err;
	    
	keydb = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keydb );

	/*  Default key must be secret and must be able to sign */
	pgpGetKeyBooleanInternal( key, kPGPKeyProperty_IsSecret, &isSecret);
	PGPValidateParam( isSecret );
	
	pgpGetKeyBooleanInternal( key, kPGPKeyProperty_CanSign, &cansign);
	PGPValidateParam( cansign );

	/* Set the default key axiomatic (note we don't require a passphrase) */
	sSetKeyAxiomatic( key, FALSE, NULL, 0, FALSE );
	
	err	= PGPGetKeyID( key, &keyID );
	if ( IsntPGPError( err ) )
	{
		PGPByte		data[ kPGPMaxExportedKeyIDSize ];
		PGPSize		exportedSize;
		
		err	= PGPExportKeyID( &keyID, data, &exportedSize );
		if ( IsntPGPError( err ) )
		{
			err	= PGPsdkPrefSetData( context,
					kPGPsdkPref_DefaultKeyID, data, exportedSize );
		}
	}
	
	return err;
}

#endif



/*  UserVal functions */

	PGPError
PGPSetKeyDBObjUserValue (PGPKeyDBObj *obj, PGPUserValue userVal)
{
	PGPValidateKeyDBObj( obj );

	pgpEnterPGPErrorFunction();

	obj->userVal = userVal;
	return kPGPError_NoErr;
}


	PGPError
PGPGetKeyDBObjUserValue (PGPKeyDBObj *obj, PGPUserValue *userVal)
{
	PGPValidateKeyDBObj( obj );

	pgpEnterPGPErrorFunction();

	*userVal = obj->userVal;
	return kPGPError_NoErr;
}

	static PGPError
pgpGetPrimaryUserIDInternal (
	PGPKeyDBObj *		 key,
	PGPAttributeType	 attributeType,
	PGPKeyDBObjRef *	 outRef)
{
	PGPKeyDBObj *	userID;

	*outRef = NULL;

	userID = pgpKeyPrimaryUserID (key, (PGPUInt32)attributeType);
	
	if( IsNull( userID ) || !pgpKeyDBObjIsReal( userID ) )
		return kPGPError_ItemNotFound;

	*outRef = userID;

	return kPGPError_NoErr;
}

	PGPError 
PGPGetPrimaryUserID (
	PGPKeyDBObj *		key,
	PGPKeyDBObjRef *	outRef)
{
	PGPValidatePtr( outRef );
	*outRef	= NULL;
	PGPValidateKey( key );

	pgpEnterPGPErrorFunction();

	return pgpGetPrimaryUserIDInternal (key, (PGPAttributeType)0, outRef );
}


	PGPError 
PGPGetPrimaryAttributeUserID (
	PGPKeyDBObj *		key,
	PGPAttributeType	 attributeType,
	PGPKeyDBObjRef *	outRef)
{
	PGPValidatePtr( outRef );
	*outRef	= NULL;
	PGPValidateKey( key );

	pgpEnterPGPErrorFunction();

	return pgpGetPrimaryUserIDInternal( key, attributeType, outRef );
}


/*
 * Get appropriate subkey (or top level key) for usage.  This mirrors the
 * logic in pgpKeyPubKey.
 */

	PGPError 
PGPGetKeyForUsage (
	PGPKeyDBObj *		key,
	PGPUInt32			usageFlags,
	PGPKeyDBObjRef *	outRef)
{
	PGPKeyDBObjRef		subkey;
	PGPByte				pkalg;

	PGPValidatePtr( outRef );
	*outRef	= NULL;
	PGPValidateKey( key );

	pgpEnterPGPErrorFunction();

	pgpAssert( OBJISTOPKEY( key ) );

	if( usageFlags & kPGPKeyPropertyFlags_UsageEncrypt )
	{
		subkey = pgpKeySubkey(key, NULL);
		if( usageFlags & kPGPKeyPropertyFlags_UsageSign )
		{
			/* Need to both sign and encrypt - must not have subkeys */
			if( IsntNull( subkey ) )
				return kPGPError_PublicKeyUnimplemented;
			pgpKeyID8 (key, &pkalg, NULL);
			if( pgpKeyAlgUse( pgpPkalgByNumber(pkalg) ) !=
								PGP_PKUSE_SIGN_ENCRYPT )
			{
				return kPGPError_PublicKeyUnimplemented;
			}
			*outRef = key;
			return kPGPError_NoErr;
		}

		/* Check that subkey can encrypt */
		if( IsNull( subkey ) )
			subkey = key;
		pgpKeyID8 (subkey, &pkalg, NULL);
		if( !( pgpKeyAlgUse( pgpPkalgByNumber(pkalg) ) & PGP_PKUSE_ENCRYPT ) )
		{
			return kPGPError_PublicKeyUnimplemented;
		}
		*outRef = subkey;
		return kPGPError_NoErr;
	}

	if( usageFlags & kPGPKeyPropertyFlags_UsageSign )
	{
		pgpKeyID8 (key, &pkalg, NULL);
		if( !( pgpKeyAlgUse( pgpPkalgByNumber(pkalg) ) & PGP_PKUSE_SIGN ) )
		{
			return kPGPError_PublicKeyUnimplemented;
		}
		*outRef = key;
		return kPGPError_NoErr;
	}
	
	return kPGPError_BadParams;
}


/*____________________________________________________________________________
	Token exported functions
____________________________________________________________________________*/

	PGPError
pgpCountTokens_internal( PGPContextRef context, PGPUInt32 *numTokens )
{
	PGPUInt32			ntoks;
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidatePtr( numTokens );
	*numTokens	= 0;
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	ntoks = pgpTokenGetCount();
	if( (PGPInt32)ntoks < 0 )
		return (PGPError) ntoks;

	*numTokens = (PGPUInt32)ntoks;
	return err;
}


	PGPError
PGPCountTokens( PGPContextRef context, PGPUInt32 *numTokens )
{
	PGPUInt32			ntoks;
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidatePtr( numTokens );
	*numTokens	= 0;
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	if( !pgpRPCEnabled() )
	{
		err = pgpCountTokens_internal( context, &ntoks );
	} else {
		err = pgpCountTokens_back( context, &ntoks );
	}
	if( (PGPInt32)ntoks < 0 )
		return (PGPError) ntoks;

	*numTokens = ntoks;
	return err;
}

	PGPError
pgpGetTokenInfo_internal( PGPContextRef context, 
                         PGPUInt32 tokNumber, PGPTokenInfo *tokenInfo)
{
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidatePtr( tokenInfo );
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	err = pgpGetTokenInfo( tokNumber, tokenInfo );
	return err;
}


   	PGPError
PGPGetTokenInfo( PGPContextRef context, PGPUInt32 tokNumber, 
                PGPTokenInfo *tokenInfo )
{
	PGPError err = kPGPError_NoErr;
	
	PGPValidatePtr( tokenInfo );
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	if( !pgpRPCEnabled() )
	{
		err = pgpGetTokenInfo_internal( context, tokNumber, tokenInfo );
	} else {
		err = pgpGetTokenInfo_back( context, tokNumber, tokenInfo );
	}

	return err;
}

	PGPError				
pgpDeleteKeyOnToken_internal( PGPContextRef context, 
		PGPUInt32 keydbID, const PGPKeyID *keyID, 
		PGPUInt32 toknumber,
		PGPByte const *pin, PGPSize pinLen )
{
	PGPKeyDBRef db = (PGPKeyDBRef)keydbID;
	PGPError		err = kPGPError_NoErr;
	PGPKeyDBObjRef	key = NULL;
	PGPByte			*keyidbytes;
	PGPToken		*tok;

	(void)context;
	(void)toknumber;

	if( !pgpKeyDBIsValid( db ) )  {
		pgpAssert(0);
		return kPGPError_MissingKeyDB;
	}

	/* Determine the top key */
	PGPFindKeyByKeyID( db, keyID, &key );
	if( !pgpKeyDBObjIsValid(key) )  {
		pgpAssert(0);
		return kPGPError_KeyInvalid;
	}

	keyidbytes = (PGPByte *)pgpGetKeyIDBytes( keyID );

	if( toknumber != -1 )  {	
		tok = pgpTokenGetIndexed( toknumber );
		pgpAssert( pgpTokenFromKeyID(keyidbytes) == tok );
	}
	else  {
		tok = pgpTokenFromKeyID(keyidbytes);
	}
	if( !tok )
		return kPGPError_SmartCardKeyNotFound;

	err = pgpTokenObjAuth( tok, (char *)pin, pinLen );
	if( IsPGPError( err ) )
		return kPGPError_BadPassphrase;

	if( ! pgpKeyIsSubkey( key ) )  {	/* Handle subkeys, if any */
		PGPKeySetRef singleKeySet = kInvalidPGPKeySetRef;
		PGPKeyIterRef iter = NULL;
		
		err = PGPNewOneKeySet( key, &singleKeySet );
		
		if( IsntPGPError( err = PGPNewKeyIterFromKeySet (
			singleKeySet, &iter ) ) )
		{
			PGPKeyDBObjRef subkey = NULL;
			
			while( IsntPGPError( PGPKeyIterNextKeyDBObj( iter,
				kPGPKeyDBObjType_SubKey, &subkey ) ) )
			{	
				PGPKeyID		subkeyID;
				PGPByte			*subkeyidbytes;
				
				pgpKeyID8 (subkey, NULL, &subkeyID );
				subkeyidbytes = (PGPByte *)pgpGetKeyIDBytes( &subkeyID );
				
				err = pgpTokenObjDeleteKey( tok, subkeyidbytes, FALSE );
				/* Ignore, if error */
			}
		}
		
		PGPFreeKeyIter( iter );

		/* Delete master key / key */
		err = pgpTokenObjDeleteKey(tok, keyidbytes, TRUE );
	}
	else  {	/* We called on subkey. Update PGP Data */
		PGPKeyDBObjRef	masterkey;

		masterkey = pgpKeyMasterkey( key );
		pgpKeyDBRemoveObject( db, key );

		/* Delete subkey */
		err = pgpTokenObjDeleteKey(tok, keyidbytes, TRUE );

		/* Update PGP Data (should not fail bacause of insufficient space) */
		err = pgpTokenCopyPubKeyToToken( context, masterkey, -1, tok );

		/* Now delete the whole key, so the client will get update */
		pgpKeyDBRemoveObject( db, masterkey );
	}

	(void)pgpTokenObjDeAuth( tok );

	pgpSyncTokenToKeyDB( context, NULL, TRUE );

	return err;
}

	PGPError
PGPDeleteKeyOnToken( PGPKeyDBObjRef key, PGPUInt32 tokNumber, 
					const PGPByte *pin, PGPSize pinLen )
{
	PGPError		err = kPGPError_NoErr;
	PGPContextRef	context;
	PGPKeyID		keyID = {{0}};
	PGPKeyDBRef		db;

	pgpAssert( pgpKeyIsOnToken(key) );
	pgpAssert( pgpKeyIsSec(key) );

	context = PGPPeekKeyDBObjContext( key );
	PGPValidateContext( context );

	PGPValidateKeyOrSubKey( key );	/* Can be called on key or subkey */
	pgpKeyID8 (key, NULL, &keyID);

	pgpEnterPGPErrorFunction();

	db = PGPPeekKeyDBObjKeyDB( key );

	if( pgpRPCEnabled() )  { 
		err = pgpDeleteKeyOnToken_back( context, 
			db->id, &keyID, tokNumber, pin, pinLen );
	}
	else  {
		err = pgpDeleteKeyOnToken_internal( context, 
			db->id, &keyID, tokNumber, pin, pinLen );
	}

	pgpKeyDBObjRefresh( key, TRUE );

	return err;
}

	PGPError
pgpWipeToken_internal( PGPContextRef context, PGPUInt32 tokNumber,
	PGPByte const *passphrase, PGPSize passphraseLength )
{
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	err = pgpTokenWipe( context, tokNumber, passphrase, passphraseLength );
	return err;
}


	PGPError
PGPWipeToken( PGPContextRef context, PGPUInt32 tokNumber,
	PGPByte const *passphrase, PGPSize passphraseLength )
{
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	if( !pgpRPCEnabled() )
	{
		err = pgpWipeToken_internal( context, tokNumber, passphrase,
									 passphraseLength );
	} else {
		err = pgpWipeToken_back( context, tokNumber, passphrase,
								 passphraseLength );
	}
	return err;
}

	PGPError
pgpTokenPassphraseIsValid_internal( PGPContextRef context, PGPUInt32 tokNumber,
	PGPByte const *passphrase, PGPSize passphraseLength )
{
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	err = pgpTokenCheckPassphrase( tokNumber, passphrase, passphraseLength );
	return err;
}


	PGPError
PGPTokenPassphraseIsValid( PGPContextRef context, PGPUInt32 tokNumber,
	PGPByte const *passphrase, PGPSize passphraseLength )
{
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	if( !pgpRPCEnabled() )
	{
		err = pgpTokenPassphraseIsValid_internal( context, tokNumber,
											passphrase, passphraseLength );
	} else {
		err = pgpTokenPassphraseIsValid_back( context, tokNumber, passphrase,
											  passphraseLength );
	}
	return err;
}

	PGPError
pgpCopyKeyToToken_internal(
	PGPKeyDBObjRef		key,
	PGPUInt32			tokNumber,
	PGPBoolean			isMaster,
	char const *		passphrase,
	PGPSize				passphraseLength,
	PGPBoolean			hashedPhrase,
	char const *		PIN,
	PGPSize				PINlength,
	PGPUInt32			cacheTimeOut,
	PGPBoolean			cacheGlobal
	)
{
	PGPError			err = kPGPError_NoErr;
	PGPContextRef		context;

	(void) cacheTimeOut;
	(void) cacheGlobal;

	pgpa(pgpaPGPKeyValid(key));
	PGPValidateKey( key );
	context = PGPPeekKeyDBObjContext( key );
	pgpAssert( tokNumber != -1 );

	err = pgpTokenCopyPrivKeyToToken( context, key, isMaster,
									  (PGPByte *) passphrase,
									  passphraseLength, hashedPhrase,
									  (PGPByte *) PIN, PINlength, tokNumber );
	if( IsntPGPError( err ) && isMaster )
	{
		err = pgpTokenCopyPubKeyToToken( context, key, tokNumber, NULL );
		if( IsPGPError( err ) )
		{
			/* Delete private part from smart card on failure */
			PGPKeyID keyID;
			pgpKeyID8( key, NULL, &keyID );
			pgpDeleteKeyOnToken( context, &keyID, tokNumber );
		}
	}

	return err;
}

static const PGPOptionType keytokenOptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase,
    kPGPOptionType_ExportKeyDBObj,
    kPGPOptionType_ExportKeySet,
    kPGPOptionType_OutputToken, 
	kPGPOptionType_ExportPrivateSubkeys
};


	PGPError
pgpCopyKeyToToken( PGPKeySet *keyset, PGPOptionListRef optionList )
{
	void *				passphrase = NULL;
	PGPSize				passphraseLength = 0;
	PGPBoolean			hashedPhrase = FALSE;
	void *				PIN = NULL;
	PGPSize				PINlength = 0;
	PGPBoolean			hashedPIN = FALSE;
	PGPUInt32			cacheTimeOut = 0;
	PGPBoolean			cacheGlobal;
	PGPError			err = kPGPError_NoErr;
    PGPKeyDBObjRef      key;

    PGPBoolean          fOutputToken=FALSE;
    PGPUInt32           tokNumber=-1;

	PGPBoolean			isSubkey = FALSE;

	pgpAssert( pgpKeySetIsValid( keyset ) );

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						keytokenOptionSet, elemsof( keytokenOptionSet ) ) ) )
		return err;

    key = pgpFirstKeyInKeySet( keyset );
    if( IsNull( key ) )  {
        err = kPGPError_BadParams;
        goto error;
    }
	pgpa(pgpaPGPKeyValid(key));

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_OutputToken, FALSE,
						 "%b%d", &fOutputToken, &tokNumber ) ) )
    pgpAssert( fOutputToken && tokNumber < 100 ); /* Sanity check */


	/* Get passphrase options */
	pgpSearchPassphraseOptions( optionList, &passphrase, &passphraseLength,
								&hashedPhrase, &PIN, &PINlength, &hashedPIN );
	if( hashedPIN )
	{
		pgpDebugMsg( "Hashed passphrase illegal for PIN" );
		err = kPGPError_BadParams;
		goto error;
	}

	/* Other optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_CachePassphrase, FALSE,
						"%d%b", &cacheTimeOut, &cacheGlobal ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_ExportPrivateSubkeys, FALSE,
						"%b", &isSubkey ) ) )
		goto error;
	

	if( pgpFrontEndKey( key ) )
	{
		err = pgpCopyKeyToToken_back( PGPPeekKeyDBObjContext(key),
								 pgpKeyDBObjID(key), tokNumber,
								 (PGPBoolean)!isSubkey, passphrase,
								 passphraseLength, hashedPhrase,
								 PIN, PINlength, cacheTimeOut, cacheGlobal );
		if( IsPGPError( err ) )
			return err;
		pgpKeyDBObjRefresh( key, FALSE );
	} else {
		err = pgpCopyKeyToToken_internal( key, tokNumber,
								 (PGPBoolean)!isSubkey, passphrase,
								 passphraseLength, hashedPhrase,
								 PIN, PINlength, cacheTimeOut, cacheGlobal );
	}

error:

	return err;
}


    PGPError
pgpTokenImportX509_internal(
	PGPContextRef	context,
    const PGPByte   *keyID, 
    const PGPByte   *userID,    PGPSize         userIDlen, 
    const PGPByte   *x509,      PGPSize         x509len,
    const PGPByte   *password,  PGPSize         passwordLength )
{
	PGPError			err = kPGPError_NoErr;
    PGPToken *tok = pgpTokenFromKeyID( keyID );

    PGPByte *subj=NULL, *issuer=NULL, *sn=NULL;
    PGPSize subj_len, issuer_len, sn_len;

    pgpAssert( tok );

    if( tok )  {
        err = pgpX509CertToDName( (PGPByte*)x509, x509len, FALSE,
            NULL, &subj_len);
        if (IsPGPError( err ))
            goto error;
        
        err = pgpX509CertToDName( (PGPByte*)x509, x509len, TRUE,
            NULL, &issuer_len);
        if (IsPGPError( err ))
            goto error;

        err = pgpX509CertToSerialNumber( (PGPByte*)x509, x509len, NULL, &sn_len );
        if (IsPGPError( err ))
            goto error;

        subj = 	(PGPByte *)pgpContextMemAlloc( context, subj_len, 0);
        issuer = (PGPByte *)pgpContextMemAlloc( context, issuer_len, 0);
        sn = (PGPByte *)pgpContextMemAlloc( context, sn_len, 0);
        if( subj == NULL || issuer == NULL || sn == NULL )
            goto error;

        err = pgpX509CertToDName( (PGPByte*)x509, x509len, FALSE,
            subj, &subj_len);
        if (IsPGPError( err ))  {
            pgpDebugMsg("pgpTokenImportX509_internal: get Subject failed");
            goto error;
        }

        err = pgpX509CertToDName( (PGPByte*)x509, x509len, TRUE,
            issuer, &issuer_len);
        if (IsPGPError( err )) {
            pgpDebugMsg("pgpTokenImportX509_internal: get Issuer failed");
            goto error;
        }

        err = pgpX509CertToSerialNumber( (PGPByte*)x509, x509len, sn, &sn_len );
        if (IsPGPError( err ))
            goto error;
        
        /* All needed X509 object are parsed, pass them to TCL */
        err = pgpTokenObjAuth( tok, (char *) password, passwordLength );
        if( IsPGPError( err ) ) {
            pgpDebugMsg("pgpTokenImportX509_internal: token authorization failed");
            goto error;
        }

        err = pgpTokenObjImportX509( tok, keyID, userID, userIDlen, 
                    x509, x509len, subj, subj_len, issuer, issuer_len, 
                    sn, sn_len );
    }

error:
    if( subj )
        PGPFreeData( subj );
    if( issuer )
        PGPFreeData( issuer );
    if( sn )
        PGPFreeData( sn );

    return err;
}

    PGPError
pgpTokenPutKeyContainer_internal(
    PGPContextRef   context,
    const PGPByte   *keyID, 
    const PGPByte   *password, PGPSize passwordSize, 
    const PGPByte   *cont, PGPSize contSize  )
{
    PGPToken *tok = pgpTokenFromKeyID( keyID );
	PGPError err;
(void) context;

    pgpAssert( tok );
    if( !tok )  
        return kPGPError_SecretKeyNotFound;

    err = pgpTokenObjAuth( tok, (const char *) password, passwordSize );
    if( IsPGPError( err ) ) {
        pgpDebugMsg("pgpTokenPutKeyContainer_internal: token authorization failed");
        return err;
    }

    return pgpTokenObjPutKeyContainer(tok, keyID, cont, contSize );
}

    PGPError
pgpTokenGetKeyContainer_internal(
    PGPContextRef   context,
    const PGPByte   *keyID, 
    const PGPByte   *password,	PGPSize passwordSize, 
    PGPByte         **contOut, PGPSize *contOutSize )
{
    PGPToken *tok = pgpTokenFromKeyID( keyID );
    PGPError ret;
    PGPSize sz = 0;
    PGPByte *c = NULL;

    pgpAssert( tok );
    if( !tok )  
        return kPGPError_SecretKeyNotFound;

    ret = pgpTokenObjAuth( tok, (const char *)password, passwordSize );
    if( IsPGPError( ret ) ) {
        pgpDebugMsg("pgpTokenGetKeyContainer_internal: token authorization failed");
        return ret;
    }

    ret = pgpTokenObjGetKeyContainer(tok, keyID, &c, &sz );

    /* Reallocate within given PGP context */
    if( sz > 0 && !IsNull(contOutSize) )  {
        PGPByte *p = pgpContextMemAlloc( context, sz, 0 );
        if( p )  {
            pgpCopyMemory( c, p, sz ); 
            pgpTokenFreeTCLMem( c );
            c = p;
        }
    }

    if( !IsNull(contOut) )
        *contOut = c;
    if( !IsNull(contOutSize) )
        *contOutSize = sz;

    return ret;
}

PGPError PGPSetPKCS11DrvFile( PGPByte *module )  
{
	pgpEnterPGPErrorFunction();

	return pgpSetPKCS11DrvFile_back( module );	
        /* calls pgpSetPKCS11DrvFile_internal if RPC is not enabled*/
}


/*____________________________________________________________________________
	Name is always returned NULL terminated.
	
	if name is null, then just the size is returned
____________________________________________________________________________*/
	PGPError
PGPGetPrimaryUserIDName(
	PGPKeyDBObjRef	key,
	void *		buffer,
	PGPSize		bufferSize,
	PGPSize *	fullLength  )
{
	PGPKeyDBObjRef		userID;
	PGPError			err	= kPGPError_NoErr;
	
	PGPValidateKey( key );
	PGPValidateParam( IsntNull( buffer ) || IsntNull( fullLength ) );
	*fullLength = 0;

	pgpEnterPGPErrorFunction();

	err	 = pgpGetPrimaryUserIDInternal (key, (PGPAttributeType)0, &userID );
	if ( IsntPGPError( err ) )
	{
		err	= pgpGetUserIDStringBuffer( userID,
			kPGPUserIDProperty_Name, bufferSize, buffer, fullLength);
	}
	
	return( err );
}

	PGPError
PGPGetPrimaryUserIDValidity (PGPKeyDBObj *key, PGPValidity *validity)
{
	PGPKeyDBObj *	userID;
	PGPError		err	= kPGPError_NoErr;

	PGPValidatePtr( validity );
	*validity	= kPGPValidity_Unknown;
	PGPValidateKey( key );

	pgpEnterPGPErrorFunction();

	err	 = pgpGetPrimaryUserIDInternal (key, (PGPAttributeType)0, &userID );
	if ( IsntPGPError( err ) )
	{
		PGPInt32	temp;
		
		err	= pgpGetUserIDNumberInternal(userID,
										 kPGPUserIDProperty_Validity, &temp);
		if ( IsntPGPError( err ) )
			*validity	= (PGPValidity)temp;
	}
	return( err );
}

	PGPContextRef
PGPPeekKeyDBContext( PGPKeyDBRef ref )
{
	pgpAssert( pgpKeyDBIsValid( ref ) );
	
	if ( ! pgpKeyDBIsValid( ref ) )
		return( kInvalidPGPContextRef );
		
	return( ref->context );
}

	PGPContextRef
PGPPeekKeyListContext( PGPKeyListRef ref )
{
	if ( ! pgpKeyListIsValid( ref ) )
		return( kInvalidPGPContextRef );
		
	return( PGPPeekKeySetContext( ref->keySet ) );
}


	PGPContextRef
PGPPeekKeySetContext( PGPKeySetRef ref )
{
	if ( ! pgpKeySetIsValid( ref ) )
		return( kInvalidPGPContextRef );
		
	return( PGPPeekKeyDBContext( ref->keyDB ) );
}

	PGPKeyDBRef
PGPPeekKeySetKeyDB( PGPKeySetRef ref )
{
	if ( ! pgpKeySetIsValid( ref ) )
		return( kInvalidPGPKeyDBRef );
		
	return( ref->keyDB );
}

	PGPContextRef
PGPPeekKeyIterContext( PGPKeyIterRef ref )
{
	if ( ! pgpKeyIterIsValid( ref ) )
		return( kInvalidPGPContextRef );
		
	if( ref->isKeyList )
		return( PGPPeekKeyListContext( ref->keyList ) );
	else
		return( PGPPeekKeySetContext( ref->keySet ) );
}


	PGPContextRef
PGPPeekKeyDBObjContext( PGPKeyDBObjRef ref )
{
	if ( ! pgpKeyDBObjIsValid( ref ) )
		return( kInvalidPGPContextRef );
		
	return( PGPPeekKeyDBContext(
		PGPPeekKeyDBObjKeyDB( ref ) ) );
}

	PGPKeyDBRef
PGPPeekKeyDBObjKeyDB( PGPKeyDBObjRef ref )
{
	if ( ! pgpKeyDBObjIsValid( ref ) )
		return( kInvalidPGPKeyDBRef );
		
	while( !OBJISTOPKEY( ref ) )
		ref = ref->up;

	return (PGPKeyDB *)ref->up;
}

	PGPKeyDBObjRef
PGPPeekKeyDBObjKey( PGPKeyDBObjRef ref )
{
	PGPKeyDBObjRef	key = kInvalidPGPKeyDBObjRef;
	
	if( pgpKeyDBObjIsValid( ref ) )
	{
		while( !OBJISTOPKEY( ref ) )
			ref = ref->up;
		key = ref;
	}

	return key;
}

	PGPKeyDBObjRef
PGPPeekKeyDBObjUserID( PGPKeyDBObjRef ref )
{
	PGPKeyDBObjRef	userID = kInvalidPGPKeyDBObjRef;
	
	if( pgpKeyDBObjIsValid( ref ) )
	{
		if( OBJISUSERID( ref ) )
		{
			userID = ref;
		}
		else if( OBJISSIG( ref ) )
		{
			userID = ref->up;
		}
		
		if( ! pgpUserIDIsValid( userID ) )
			userID = kInvalidPGPKeyDBObjRef;
	}

	return userID;
}


/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
